<?php

/*
 * pomocne funkce pro zobrazovani view a jeho skladani z casti
 */

namespace PHPView;

use IZON\DBLocale\Interceptor\LocaleInterceptor;
use IZON\IO\Image;
use IZON\MVC\FlashMessage;
use IZON\MVC\PageInfo;
use IZON\MVC\Views\PHPLayoutView;
use IZON\MVC\Views\PHPView;
use IZON\Utils\Locale;

/**
 * inserts layout part to specified place of view
 * @param string $partName jak se dana cast jmenuje
 * @param bool $required jetli musi byt primo v danem adresari sablony
 */
function insertLayoutPart(string $partName, bool $required = false)
{
    /** @var PHPLayoutView $view */
    $view = PHPView::getApplicationView();
    $view->renderLayoutPart($partName, $required);
}

/**
 * returns parameter with name $paramName passed to view vrom controller in ModelAndView
 * @param string $paramName
 * @return mixed value of parameter of null if not exists
 */
function getParameter($paramName)
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->getParameter($paramName);
}

/**
 * returns true if $paramName was passed to view
 * @param string $paramName
 * @return boolean
 */
function hasParameter($paramName)
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->hasParameter($paramName);
}

/**
 * Vrati \IZON\MVC\FlashMessage na zaklade jeji id a vymaze ji z registru<br>
 * Pokud nebude id zadano, veme se prvni ze seznamu zprav a seznam se komplet vyprazdni
 *
 * @param null|string $id
 * @return  FlashMessage|null
 */
function getFlashMessage(?string $id = null)
{
    $ret = null;
    $context = getFlashMessagesContext();
    // otevri session jestli jeste nebyla otevrena
    $sessionNotStarted = session_status() == PHP_SESSION_NONE;
    if ($sessionNotStarted) { // jeste nebyla nastartovana sessiona, nastartovat
        // server nemusi byt schopen nastavit ini_set('session.use_cookies', 0), aby se cookie se session id posilalo jen jednou pri initu aplikace
        @session_start();
    }

    if (is_null($id)) {
        $swap = getFlashMessages();
        $ret = reset($swap);
        unset($_SESSION['flashMessages'][$context]);
    } elseif (array_key_exists($id, getFlashMessages())) {
        $ret = getFlashMessages()[$id];
        unset($_SESSION['flashMessages'][$context][$id]);
    }

    if ($sessionNotStarted) { // pred timto otevrenim session jeste nebyla session nastartovana, tak je ji mozno zavrit
        session_write_close();
    }

    return $ret;
}

/**
 * Zjisti, jestli je zaregistrovana nejaka (v pripade zadaneho id konkretni) zprava k zobrazeni
 *
 * @param null|string $id
 * @return bool
 */
function hasFlashMessage(?string $id = null)
{
    if (is_null($id)) {
        return !empty(getFlashMessages());
    }
    return array_key_exists($id, getFlashMessages());
}

/**
 * Return context of flash messages
 *
 * @return string [web|admin]
 */
function getFlashMessagesContext(): string
{
    $context = 'web';
    if (array_key_exists('adminLoggedAdminUser', $_SESSION)) {
        $context = 'admin';
    }
    return $context;
}

/**
 * Vrati vescky zaregistrovane flashmessages
 *
 * @return array
 */
function getFlashMessages()
{
    $context = getFlashMessagesContext();
    if (isset($_SESSION['flashMessages'][$context])) {
        return (array)$_SESSION['flashMessages'][$context];
    }
    return [];
}

/**
 * returns url
 * @param string $relativeURL
 * @deprecated
 */
function getURL($relativeURL)
{
    return "/" . $relativeURL;
}

/**
 * @param string $relativeURL
 * @deprecated
 * returns url with current timestamp
 */
function getTimeStampedURL(string $relativeURL): string
{
    $fsPath = __BASE_DIR__ . "/www/" . $relativeURL; // @phpstan-ignore-line
    $fileTime = time();
    if (file_exists($fsPath)) { // try to find file
        $fileTime = filemtime($fsPath);
    }
    return getURL($relativeURL) . "?ts=" . $fileTime;
}

/**
 * vrati odkaz odpovidajici controlleru se zadanimy parametry
 * @param string $controllerId id controlleru z DI containeru, ktery se ma zaolat
 * @param array $params parametry, potrebne k doplneni controleru
 * @param string $methodName metoda controlleru, ktera se ma zavolat
 * @param string|Locale $locale locale string used to define which locale version of web to show
 */
function getControllerURL($controllerId, array $params = [], $methodName = null, $locale = null)
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    if ($locale == null) {
        $locale = $view->getLocale();
    } elseif (is_string($locale)) {
        $locale = Locale::forLocaleTag($locale);
    }
    //    var_dump($locale);
    return $view->getRouter()->findURL($controllerId, $params, $methodName, $locale);
}

/**
 * formats number for better number view
 * @param float $amount
 *
 * TODO: pridat fomatovani na zaklade locale
 */
function formatNumber($amount, ?Locale $locale = null): string
{
    $round = 0;
    // TODO: pridat rounding
    $str = number_format($amount, $round, ',', '&nbsp;');
    return $str;
}

/**
 * returns locaje class containing language and country part
 * @return Locale
 */
function getLocale()
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    return $view->getLocale();
}

/**
 * @return Locale
 * @deprecated use getLocale() instead
 */
function getLocaleEx()
{
    return getParameter(LocaleInterceptor::AVAILABLE_AS); // @phpstan-ignore-line
}

/**
 * Render js a css
 * @param array $data
 * @param string $type [js|css]
 * @param string $context [web|admin]
 * @deprecated use includeScript/includeScripts and includeStyle/includeStyles instead
 *
 * $data = either one dimensional field with paths to files, or<br>
 * field fields, where nested fierld has structure<br>
 * $script[external] - jedna se o externi js/css - pokud neni, netreba zadavat<br>
 * $script[src] - url, ze ktere se ma script natahnout<br>
 * pro css je jeste dostupny $script[media] - pro jaky typ media budou css pouzita - nepovinny parametr
 * <br><br>
 * For example: $data = [['external'=>true, 'src' => 'http://www.something-somewher.com/script.js']]<br>
 * $data = ['js/app.js', 'js/common.js', 'js/jquery.js']<br>
 *
 *
 */
function insertScripts(array $data, $type, $context = 'web')
{
    foreach ($data as $key => $value) {
        if (empty($value)) {
            continue;
        }
        // little bit hack for admin css and js files
        if (!is_array($value) && $context == 'admin') {
            $value = ['src' => $value];
        }

        if (!is_array($value)) {
            $value = explode("?", $value)[0];
            if ($type == 'js') {
                includeScript("/" . $value);
            } elseif ($type == 'css') {
                includeStyle("/" . $value);
            }
        } else {
            if ($value['external']) {
                if ($type == 'js') {
                    echo '<script type="text/javascript" src="' . $value['src'] . '"></script>' . PHP_EOL;
                } elseif ($type == 'css') {
                    $media = $value['media'] ?? '';
                    echo '<link href="' . $value['src'] . '" rel="stylesheet" type="text/css"' . ($media
                        != '' ? ' media="' . $media . '"' : '') . ' />' . PHP_EOL;
                }
            } else {
                $src = explode("?", $value['src'])[0];
                if ($type == 'js') {
                    includeScript("/" . $src);
                } elseif ($type == 'css') {
                    includeStyle("/" . $src);
                    //                        echo '<link href="' . $urlFunction($value['src']) . '" rel="stylesheet" type="text/css"' . ($value['media'] != '' ? ' media="' . $value['media'] . '"' : '') . ' />' . LF;
                }
            }
        }
    }
}

function includeScript($scriptURL, $type = "js")
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    $staticFileURL = $view->getStaticFileURL($scriptURL);
    echo '<script type="text/javascript" src="' . $staticFileURL . '"></script>' . PHP_EOL;
}

function includeScripts(array $scripts, $type = "js")
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    foreach ($scripts as $script) {
        $staticFileURL = $view->getStaticFileURL($script);
        echo '<script type="text/javascript" src="' . $staticFileURL . '"></script>' . PHP_EOL;
    }
}

function includeStyle($style, $type = "css", $params = [])
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    $staticFileURL = $view->getStaticFileURL($style);
    echo '<link href="' . $staticFileURL . '" rel="stylesheet" type="text/css" ' . (($params['media'] ?? '') != '' ? ' media="' . $params['media'] . '"' : '') . ' />' . PHP_EOL;
}

function includeStyles(array $styles, $type = "css", $params = [])
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    foreach ($styles as $style) {
        $staticFileURL = $view->getStaticFileURL($style);
        echo '<link href="' . $staticFileURL . '" rel="stylesheet" type="text/css" ' . (($params['media'] ?? '') != '' ? ' media="' . $params['media'] . '"' : '') . ' />' . PHP_EOL;
    }
}

/**
 * inserts file content to view where function was called
 * @param string $filePath
 */
function insertFileContent($filePath)
{
    /** @var PHPView $view */
    $view = PHPView::getApplicationView();
    $staticFileURL = $view->getStaticFileURL($filePath);
    return file_get_contents($staticFileURL);
}

/**
 * Retruns \IZON\MVC\PageInfo object
 *
 * @return null|PageInfo
 */
function getPageInfo()
{
    $pageInfo = getParameter(PageInfo::PAGE_INFO_INDENTIFIER);
    return $pageInfo;
}

/**
 * Check if $controllerId is current page processing controller
 * @param string $controllerId to check
 * @return bool True - yes is actual False - not actual
 */
function isControllerActualPageController($controllerId)
{
    /** @var PageInfo $pageInfo */
    $pageInfo = getParameter(PageInfo::PAGE_INFO_INDENTIFIER);
    if ($pageInfo->getControllerId() == $controllerId) {
        return true;
    }
    return false;
}

/**
 * Clear string from diacritics and other "dangerous" characters
 * @param string $string
 * @return string
 * @deprecated use SlugUtils::createSlug() instead
 */
function clearString($string)
{
    return transliterator_transliterate('Any-Latin; Latin-ASCII; [\u0100-\u7fff] remove', $string);
}


/// backward compatibility
/**
 * @param Image|string $sourceImage source Image Object or path to image file with first "/" path_to_file
 * @param array $parameters parameters for resizing of image, supported settings:<br>
 * width => integer, <br>height => integer, <br>type => [cover|contain], default is contain, <br>quality = number, default = 90<br>
 * for example: ["width" => 1234, "height" => 1234, "type" => "contain"]
 *
 * @return string
 * @deprecated use PHPView\Thumber\getThumbnail
 * Return image url for source path or Image object
 */
function getThumbnail($sourceImage, array $parameters = [])
{
    return Thumber\getThumbnail($sourceImage, $parameters); // @phpstan-ignore function.notFound
}

/**
 * @param Image|string $sourceImage source Image Object or path to image file with first "/"
 * @param array $parameters parameters for resizing of image, supported settings:<br>
 * width => integer, <br>height => integer, <br>type => [cover|contain], default is contain, <br>quality = number, default = 90, max is 100 - create image without any compression<br>
 * for example: ["width" => 1234, "height" => 1234, "type" => "contain"]
 * @return string
 * @deprecated use PHPView\Thumber\getThumb
 */
function getThumb($sourceImage, array $parameters = [])
{
    return Thumber\getThumbnail($sourceImage, $parameters); // @phpstan-ignore function.notFound
}

/**
 * @param string $name svg file name
 * @param string $path path to svg file, default is htdocs/www/images/svg
 * @return string
 * @deprecated use insertFileContent instead
 * Returns svg file content
 *
 */
function insertSvg($name, $path = 'www/images/svg')
{
    return file_get_contents(__BASE_DIR__ . '/' . $path . '/' . $name . '.svg'); // @phpstan-ignore-line
}
