<?php

namespace IZON\MVC\Routers;

use Exception;
use IZON\MVC\Exceptions\PageNotFoundException;
use IZON\MVC\HttpRequest;
use IZON\MVC\Interceptors\Interceptor;
use IZON\MVC\Locale\LocaleResolver;
use IZON\Utils\Locale;
use IZON\Utils\StringUtils;

/**
 * finds controller findRoute for HttpRequest
 * finds url findURL for $controllerId
 * url => controller and controllerId => url translations are defined in $urlRouteDefinitions passed in constructor
 */
class Router
{
    /**
     * @var RouteDefinition[] definitions of routes
     */
    protected $urlRouteDefinitions = [];

    /**
     *
     * @var string[]|Interceptor[] interceptors to  be called for each request resolved by this router
     */
    protected $commonInterceptors = [];

    /**
     * @var LocaleResolver
     */
    protected $localeResolver;

    /**
     * @var boolean pokud je true tak pro stranky s koncovym lomitkem v url provede externi presmervani na stranku bez neho pokud takove stranky existuji
     */
    protected $redirectToURLWithoutTailingSlash = true;


    public function __construct(array $urlRouteDefinitions)
    {
        $this->urlRouteDefinitions = $urlRouteDefinitions;
    }

    /**
     * pro request najde informace, jaky controller se ma volat a jake interceptory
     * @param HttpRequest $httpRequest
     * @return RouteInfo
     * @throws Exception
     */
    public function findRoute(HttpRequest $httpRequest)
    {
        $url = $httpRequest->getURL();
        // removes tailing /
        if (
            $this->redirectToURLWithoutTailingSlash
            && StringUtils::endsWith($url, "/")
            && $url != '/'
        ) {
            $urlWithoutSlash = mb_substr($url, 0, mb_strlen($url) - 1); // urizni koncove lomitko
            $newRequest = new HttpRequest($urlWithoutSlash, $httpRequest->getParameters(), $httpRequest->getLocale());

            // try to find route for url without tailing /
            foreach ($this->urlRouteDefinitions as $definition) {
                $routeInfo = $definition->findRoute($newRequest);
                if ($routeInfo != null) {
                    // add locale data to url
                    $newUrl = $this->localeResolver->modifyOutpultURL($urlWithoutSlash, $newRequest->getLocale());
                    $routeInfo = new RouteInfo();
                    $routeInfo->setRedirectCode(true);
                    $routeInfo->setURL($newUrl);

                    // send redirect to new address
                    return $routeInfo;
                }
            }
        }

        // try to find address as is
        foreach ($this->urlRouteDefinitions as $definition) {
            $routeInfo = $definition->findRoute($httpRequest);
            if ($routeInfo != null) {
                // prida spolecne interceptory na zacatek pole
                $interceptors = $this->commonInterceptors;
                $interceptors = array_merge($interceptors, $routeInfo->getInterceptors());
                $routeInfo->setInterceptors($interceptors);

                // prepare context
                $context = $routeInfo->getContext();
                if ($context !== null) {
                    $context->setLocale($httpRequest->getLocale());
                    $context->setRouter($this);
                }

                return $routeInfo;
            }
        }

        // no routing find
        throw new PageNotFoundException(
            "Routing of address " . $httpRequest->getURL()
            . " with parameters: " . print_r($httpRequest->getParameters(), true)
            . " not found."
        );
    }

    /**
     * pro controller s id $controllerId najde adresu pres kterou je ho mozno volat
     * @param string $controllerId id controlleru
     * @param array $parameters parametry, ptere se maji predat do adresy
     * @param null|string $methodName nazev metody, ktera se ma na controlleru zavolat
     * @param Locale $locale
     * @return string
     * @throws Exception
     */
    public function findURL($controllerId, array $parameters, $methodName, $locale)
    {
        foreach ($this->urlRouteDefinitions as $definition) {
            $url = $definition->findURL($controllerId, $parameters, $methodName, $locale);
            if ($url != null) {
                $url = $this->localeResolver->modifyOutpultURL($url, $locale);
                return $url;
            }
        }
        throw new Exception("URL for controller $controllerId and parameters " . print_r($parameters, true) . " not found");
    }

    public function setCommonInterceptors(array $commonInterceptors)
    {
        $this->commonInterceptors = $commonInterceptors;
    }

    public function setLocaleResolver(LocaleResolver $localeResolver)
    {
        $this->localeResolver = $localeResolver;
    }

    public function setRedirectToURLWithoutTailingSlash($redirectToURLWithoutTailingSlash)
    {
        $this->redirectToURLWithoutTailingSlash = $redirectToURLWithoutTailingSlash;
    }
}
