<?php

namespace IZON\MVC\Routers;

use Exception;
use IZON\MVC\Context\AjaxContext;
use IZON\MVC\Exceptions\PageNotFoundException;
use IZON\MVC\HttpRequest;
use IZON\MVC\Interceptors\PageInfoInterceptor;
use IZON\Utils\Locale;
use IZON\Utils\StringUtils;

/**
 * refinuje routovani pro ajaxive dotazy
 */
class AjaxRouteDefinition implements RouteDefinition
{
    /**
     * interceptory vazane na jednotlive controllery
     * pole poli, klic je uid controlleru
     * a hodnota je pole s id jednotlivych controlleru
     * @var array<string, string[]>
     */
    protected array $controllerInterceptors = [];

    /**
     * interceptory pridavane pred vsechny controllery teto oute definition
     * @var string[]
     */
    protected array $interceptors = [];

    /**
     * @var string jak maji zacinat funkce, ktere se pouzivaji pro ajax
     */
    protected string $methodPrefix = "ajax";


    public function findRoute(HttpRequest $request)
    {
        if (!StringUtils::startsWith($request->getURL(), "/_ajax/")) { // neobsluhuje se timto routerem
            return null;
        }

        if (
            preg_match(
                '#^/_ajax/([A-Za-z0-9.]+)/([A-Za-z0-9]+)$#',
                $request->getURL(),
                $matches
            )
        ) {  // reqexp musi byt obalen delimiterem
            $routeInfo = new RouteInfo();
            $routeInfo->setURL($request->getURL());
            $routeInfo->setContrlollerId($matches[1]);
            $routeInfo->setMethodName($matches[2]);
            $routeInfo->setParameters($request->getParameters());

            $interceptors = $this->interceptors;
            if (isset($this->controllerInterceptors[$matches[1]])) {
                $interceptors = array_merge($interceptors, $this->controllerInterceptors[$matches[1]]);
            }
            $routeInfo->setInterceptors($interceptors);

            $context = new AjaxContext($request->getSession());
            $routeInfo->setContext($context);

            // nastavuje id volaneho controlleru
            $request->setCalledControllerId($matches[1]);

            return $routeInfo;
        } else {
            throw new PageNotFoundException(
                "Url " . $request->getURL() . " musi byt ve tvaru /_ajax/controllerId/methodName"
            );
        }
    }

    /**
     * nastavuje interceptory platne pro vsechny controllery z teto route definition
     * @param array $interceptors
     */
    public function setInterceptors(array $interceptors)
    {
        $this->interceptors = array_merge(
            [new PageInfoInterceptor()], // pridava PageInfoInterceptor ktery nacit informace o prave zobrazovane strance
            $interceptors
        );
    }

    public function findURL($controllerId, array $parameters, $methodName, Locale $locale)
    {
        if (empty($methodName) || !StringUtils::startsWith($methodName, $this->methodPrefix)) {
            return null;
        }

        $outputURL = "/_ajax/" . $controllerId . "/" . $methodName;

        $query = http_build_query($parameters);
        if ($query !== "") {
            $outputURL .= "?" . $query;
        }
        return $outputURL;
    }

    /**
     * nastavuje interceptory jen pro specificke controllery
     * @param array $interceptors pole poli kde klicem je identifier controlleru a hodnota je pole interceptoru, ktere se na nem maji provezt
     */
    public function setControllerInterceptors(array $interceptors)
    {
        $this->controllerInterceptors = $interceptors;
    }

    /**
     * adds ides of interceptors to be called around $controllerId
     * @param string $controllerId
     * @param string[] $interceptorIdes
     * @throws Exception
     */
    public function addControllerInterceptors($controllerId, array $interceptorIdes)
    {
        if (array_key_exists($controllerId, $this->controllerInterceptors)) {
            throw new Exception("Interceptors for controller $controllerId already added.");
        }
        $this->controllerInterceptors[$controllerId] = $interceptorIdes;
    }
}
