<?php

namespace IZON\MVC;

use Exception;
use IZON\DI\ContainerBuilder;
use IZON\Logs\Logger;
use IZON\MVC\Exceptions\Handlers\InternalServerErrorExceptionHandler;
use IZON\MVC\Routers\Router;
use IZON\Utils\StringUtils;
use Psr\Container\ContainerInterface;
use function IZON\DI\MVC\appSubDir;

/**
 * slouzi k nacteni konfigurace
 */
class Config
{
    /**
     * pod jakym identifikatorem je ulozen router
     */
    public const ROUTER_IDENTIFIER = 'mvc.router';

    /**
     * pod jakym identifikatorem jsou ulozeny spolecne interceptory pro vsechny route definition
     */
    public const ROUTER_COMMON_INTERCEPTORS_IDENTIFIER = 'mvc.router.commonInterceptors';

    /**
     * jsak jsou definovany routery
     */
    public const ROUTER_DEFINITIONS_INDENTIFIER = 'mvc.routeDefinitions';

    /**
     * if remove tail slashes from adresses automaticaly
     */
    public const ROUTER_REDIRECT_RO_URL_WITHOUT_TAILING_SLASH_INDENTIFIER = 'mvc.router.redirectToURLWithoutTailingSlash';

    /**
     * pod jaky identifilatorem je ulozen locale resolver
     */
    public const LOCALE_RESOLVER_IDENTIFIER = 'locale.resolver';

    /**
     * provede inicializaci loggeru
     */
    public const INIT_LOGGER_IDENTIFIER = 'logs.initLogger';

    /**
     * pod jakym id jsou view resolvery
     */
    public const VIEW_RESOLVER_IDENTIFIER = 'mvc.view.viewResolvers';

    /**
     * jsak jsou definovany exception handlery
     */
    public const EXCEPTION_HANDLERS_INDENTIFIER = 'mvc.exceptionHandlers';

    /**
     * pod jakym idrentifikatorem je ulozan koremovy adresar aplikace
     */
    public const APP_DIR_IDENTIFIER = 'app.baseDir';

    /** @deprecated use APP_DIR_IDENTIFIER instead */
    public const APP_DIR = 'app.baseDir';

    /**
     * pod jakym idrentifikatorem je ulozan adresar s konfiguracenmi aplikace
     */
    public const CONFIG_DIR_IDENTIFIER = 'app.configDir';

    /**
     * pod jakym idrentifikatorem je ulozan tmp adresar aplikace
     */
    public const TMP_DIR_IDENTIFIER = 'app.tmpDir';

    /**
     * pod jakym idrentifikatorem je ulozan adresar cache aplikace
     */
    public const CACHE_DIR_IDENTIFIER = 'app.cacheDir';

    /**
     * urcuje jestli se maji cachovat konfigy: hlavni, moduly a obsah adresate web
     */
    public const USE_CONFIG_CACHE_IDENTIFIER = 'cache.useDefinitionCache';

    /**
     * do jakeho souboru se maji zapsat informce o nainstalovanych knihovnach aplikace
     */
    public const MODULE_CONFIG_FILENAME = "installed-modules.php";
    /** @deprecated use MODULE_CONFIG_FILENAME */
    public const INSTALLED_LIBS_FILENAME = "installed-modules.php";

    /**
     * @var bool jestli je vyvojovy server
     */
    protected static bool $develserver = false;

    /**
     * @var ContainerInterface|array docasna konfigurace, aby bylo mozno ji vytahnout
     */
    protected static $extConfig;

    /**
     * v jakem adresari se nachazi cela aplikace
     * @var string
     */
    protected string $baseDir;

    /**
     * klic pod kterym je ulozeno macineid
     * @var string
     */
    protected string $machineIdKey = 'MACHINE_ID';

    /**
     * v jakem akresari se nachazeji konfigy
     * @var string
     */
    protected string $configDir = 'config';

    /**
     * jaky stringem zacina konfiguracni soubor
     * @var string
     */
    protected string $configFilePrefix = "config";

    /**
     * v jakem adresari se nachazeji docasne soubory
     * @var string
     */
    protected string $tmpDir = 'app-data/tmp';

    /**
     * @var string v jakem adresari se nachazi cache aplikace
     */
    protected string $cacheDir;

    /**
     * jak se jmenuje soubor s kesi definic
     * @var string
     */
    protected string $configCacheFileName = "config-cache.php";

    /**
     * container builder slouzici k vytvoreni containeru
     * @var class-string<ContainerBuilder> ContainerBuilder
     */
    protected string $containerBuilderClass;

    /**
     * @var null|ContainerInterface DI container z ktereho je mozne vytahovat controllery
     */
    protected ?ContainerInterface $container = null;

    /**
     * @var array containing definitions of di container
     */
    protected array $definitionsArray = [];

    protected ?Logger $log = null;

    public function __construct(
        $baseDir,
        $containerBuilderClass = ContainerBuilder::class
    ) {
        $this->baseDir = $baseDir;
        $this->containerBuilderClass = $containerBuilderClass;

        $this->cacheDir = $this->tmpDir . "/cache";
    }

    public static function isDevelServer()
    {
        return self::$develserver;
    }

    /**
     * vrati hodnotu parametru z konfigu
     *
     * @param string $param Entry name
     * @return array|bool|float|int|object|string
     */
    public static function getValue($param)
    {
        if (is_array(self::$extConfig)) {
            return self::$extConfig[$param];
        } else {
            return self::$extConfig->get($param);
        }
    }

    /**
     * initializes config
     * @throws Exception
     */
    public function init()
    {
        $containerBuilderClass = $this->containerBuilderClass;
        $builder = new $containerBuilderClass();

        // nastavuje zakladni cesty v aplikaci
        $pathDefinitions = [
            self::APP_DIR_IDENTIFIER => $this->baseDir,
            self::CONFIG_DIR_IDENTIFIER => appSubDir("/" . $this->configDir),
            self::TMP_DIR_IDENTIFIER => appSubDir("/" . $this->tmpDir),
            self::CACHE_DIR_IDENTIFIER => appSubDir("/" . $this->cacheDir),
        ];
        $builder->addDefinitions($pathDefinitions);

        // nastavuje konfig
        $configDir = $this->getAbsoluteConfigDir();

        // nacte zakladni konfiguraci
        $rootConfig = $configDir . "/" . $this->configFilePrefix . ".php";
        if (!file_exists($rootConfig)) {
            throw new Exception("Neexistuje základní konfig $rootConfig");
        }
        $builder->addDefinitions($rootConfig);

        // docasny konfig dokud se nevytvori di kontainer
        $tempConfig = require $rootConfig;

        // nacte specificke konfigurace podle MACHINE_ID
        if (getenv($this->machineIdKey) != '') {
            //            $this->log->info( $this->machineIdKey ." is set to ". getenv($this->machineIdKey));

            $machineinfo = explode('-', getenv($this->machineIdKey));

            for ($i = 1; $i <= count($machineinfo); $i++) {
                $tmp = '';
                for ($j = 0; $j < $i; $j++) {
                    $tmp .= '-' . $machineinfo[$j];
                }
                $filePath = $configDir . "/" . $this->configFilePrefix . $tmp . '.php';
                if (file_exists($filePath)) {
                    //                    $this->log->info( "Adding config ". $filePath);

                    // nacte specifickou konfiguraci
                    $builder->addDefinitions($filePath);

                    // slit s korenovou konfiguraci
                    $tempConfig = array_merge($tempConfig, require $filePath);
                }
            }
        }

        // nastavuje develserver podle konfigu
        self::$develserver = $tempConfig["develserver"];
        self::$extConfig = $tempConfig;

        // nacist ostatni konfigurace webu
        $webDir = $this->getAbsoluteConfigDir() . "/web";
        foreach (scandir($webDir) as $file) {
            if (StringUtils::endsWith($file, ".php")) {
                $builder->addDefinitions($webDir . "/" . $file);
            }
        }
        // vytvori di container
        $this->container = $builder->build();
        self::$extConfig = $this->container;

        // incializuje logger
        $this->container->get(self::INIT_LOGGER_IDENTIFIER);

        // sprovoznit logovani
        $this->log = Logger::getLogger(self::class);
    }

    /**
     * vrati absolutni odkaz do nofiguracniho souboru
     * @return string
     */
    protected function getAbsoluteConfigDir()
    {
        return $this->baseDir . "/" . $this->configDir;
    }

    /**
     *
     * @return null|ContainerInterface
     */
    public function getContainer(): ?ContainerInterface
    {
        return $this->container;
    }

    /**
     * vrari router pro aplikaci
     * @return Router
     * @throws Exception
     */
    public function getRouter(): Router
    {
        $container = $this->container;
        if ($container->has(self::ROUTER_IDENTIFIER)) {
            $router = $container->get(self::ROUTER_IDENTIFIER);
        } else {
            if (!$container->has(self::ROUTER_DEFINITIONS_INDENTIFIER)) {
                throw new Exception(
                    "V konfuguraci musis byt nastaveny definice routovani adres v klici 'mvc.routeDefinitions'"
                );
            }
            $router = new Router($container->get(self::ROUTER_DEFINITIONS_INDENTIFIER));
        }
        if ($container->has(self::ROUTER_COMMON_INTERCEPTORS_IDENTIFIER)) {
            $router->setCommonInterceptors($container->get(self::ROUTER_COMMON_INTERCEPTORS_IDENTIFIER));
        }

        /** @var Router $router */
        $router->setLocaleResolver($this->getLocaleResolver());
        if ($container->has(self::ROUTER_REDIRECT_RO_URL_WITHOUT_TAILING_SLASH_INDENTIFIER)) {
            $router->setRedirectToURLWithoutTailingSlash(
                $container->get(self::ROUTER_REDIRECT_RO_URL_WITHOUT_TAILING_SLASH_INDENTIFIER)
            );
        }

        if (!$container->has(self::ROUTER_IDENTIFIER)) { // ulozit router do id pokud tam neni
            if (!method_exists($container, 'set')) {
                throw new Exception("Cant set router to container set() method is not implemented in container");
            }
            $container->set(self::ROUTER_IDENTIFIER, $router);
        }

        return $router;
    }


    // TODO: cesty vyhodit do promennych tridy

    public function getLocaleResolver()
    {
        return $this->container->get(self::LOCALE_RESOLVER_IDENTIFIER);
    }

    public function getViewResolvers()
    {
        $container = $this->container;
        if (!$container->has(self::VIEW_RESOLVER_IDENTIFIER)) {
            throw new Exception(
                "V konfuguraci musis byt nastaveny definice routovani adres v klici 'mvc.routeDefinitions'"
            );
        }
        return $container->get(self::VIEW_RESOLVER_IDENTIFIER);
    }

    /**
     * returns exception handlers for application
     */
    public function getExceptionHandlers()
    {
        $container = $this->container;
        if ($container->has(self::EXCEPTION_HANDLERS_INDENTIFIER)) {
            $handlers = $container->get(self::EXCEPTION_HANDLERS_INDENTIFIER);
            $this->log->info("Using config exception handlers");
        } else {
            // use default exception handlers
            $handlers = [
                new InternalServerErrorExceptionHandler($this->baseDir . "/www/errors/500.php")
            ];

            $this->log->info("Using default exception handlers");
        }

        return $handlers;
    }

    /**
     * @return Logger returns aplication level logger
     */
    public function getAppLogger()
    {
        return Logger::getLogger(App::class);
    }

    /**
     * returns all keys defined in config
     * @return array
     */
    public function getDefinedKeys()
    {
        return array_keys($this->definitionsArray);
    }

    public function getConfigDir()
    {
        return $this->configDir;
    }

    public function getCacheDir()
    {
        return $this->cacheDir;
    }

    /**
     * vrati basedir pro aplikaci
     * @return string
     */
    public function getBaseDir()
    {
        return $this->baseDir;
    }
}
