<?php

namespace IZON\MVC;

use Exception;
use IZON\Logs\Logger;
use IZON\MVC\Context\Context;
use IZON\MVC\Context\WebContext;
use IZON\MVC\Exceptions\Handlers\AppExceptionHandlerInterface;
use IZON\MVC\Exceptions\Handlers\InternalServerErrorExceptionHandler;
use IZON\MVC\Interceptors\PreHandleResult;
use IZON\MVC\Locale\LocaleResolver;
use IZON\MVC\Routers\RouteInfo;
use IZON\MVC\Routers\Router;
use IZON\MVC\Views\Resolvers\ViewResolver;
use IZON\Utils\StringUtils;
use Psr\Container\ContainerInterface;
use ReflectionClass;
use ReflectionMethod;
use ReflectionNamedType;
use Throwable;

/**
 * aplikace
 */
class App
{
    /**
     * @var string directory app is stored in
     */
    protected string $baseDir;

    /**
     * konfugurace aplikace
     * @var Config
     */
    protected Config $config;

    /**
     * logger pro logovani prubeho vykonavani aplikace
     * @var Logger
     */
    protected Logger $log;

    /**
     * DI container z ktereho je mozne vytahovat controllery
     * @var null|ContainerInterface
     */
    protected ?ContainerInterface $container = null;

    /**
     * router, ktery podle url vrati ktery controller se ma zavoloat
     * @var Router
     */
    protected Router $urlRouter;

    /**
     * zjisti jake locale se ma pouzit
     * @var LocaleResolver
     */
    protected LocaleResolver $localeResolver;

    /**
     * slouzi k ziskani view, ktere se ma vyrenderovat
     * @var ViewResolver[]
     */
    protected array $viewResolvers;

    /**
     * resolvery pro zpracovani vyjimek vyhozenych systemech
     * @var AppExceptionHandlerInterface[]
     */
    protected array $exceptionHandlers = [];

    /**
     * relative Path to file that require on init app error
     * @var string
     */
    protected string $initErrorPage;

    /**
     * App constructor.
     * @param Config|string $baseDir
     * @param class-string<Config> $configClass
     */
    public function __construct(
        $baseDir,
        string $configClass = Config::class
    ) {
        // sets basedir of aplication for easier url handling
        // TODO: potom nabradir necim co se jednoduseji spravuje

        if ($baseDir instanceof Config) {
            $this->baseDir = $baseDir->getBaseDir();
            $this->config = $baseDir;
        } elseif (is_string($baseDir)) {
            $this->baseDir = $baseDir;
            $this->config = new $configClass($baseDir);
        } else {
            throw new Exception("Base dir must be string or instance of " . Config::class);
        }

        /**
         * @deprecated will be removed in 4.3 use
         */
        define("__BASE_DIR__", $this->baseDir);

        $this->initErrorPage = $baseDir . '/www/errors/500.php';
    }

    /**
     * spusti zpracocani dotazu db dotazu
     */
    public function dispatch()
    {
        $routeInfo = null;
        try {
            // vytvori a nastavi request
            $httpRequest = $this->populateHTTPRequest($this->container);
            // vytvori session id a posle ho prohlizeci, potom session odemce
            $httpRequest->getSession()->init();

            // TODO: asi vydejit obsluhu presmerovani nekam ma jedno misto
            // najde routovani
            $routeInfo = $this->urlRouter->findRoute($httpRequest);
            if ($routeInfo->getRedirectCode() !== null) {
                // do redirect
                $this->log->info("redirect from router to " . $routeInfo->getURL());
                header("Location: " . $routeInfo->getURL());
                // ukoncit vykonavani
                return;
            }

            $this->log->info("Routing url " . $httpRequest->getURL() . " to " . $routeInfo->getContrlollerId());

            // najde interceptory
            $interceptors = $routeInfo->getInterceptors();
            // if interceptor specified by identifier, replace by interceptor from container
            for ($i = 0; $i < count($interceptors); $i++) {
                $interceptor = $interceptors[$i];
                if (is_string($interceptor)) {
                    $this->log->info("Reatriving interceptors " . $interceptor . " from container");
                    $interceptors[$i] = $this->container->get($interceptor);
                }
            }

            $this->log->info("Retrieved interceptors for controller " . $routeInfo->getContrlollerId() . " from container");

            // zavola preHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $this->log->info("Calling preHandle of interceptor " . get_class($interceptor) . "");
                $preHandleResult = $interceptor->preHandle($httpRequest);

                if ($preHandleResult instanceof PreHandleResult) { // ma se provect presmerovani
                    $redirectURL = $preHandleResult->getRedirectAddress();
                    if ($preHandleResult->isInternalRedirect()) {
                        $redirectControlerId = $preHandleResult->getControllerId();
                        // najde url pro presmerovani
                        $methodName = $preHandleResult->getMethodName();
                        $locale = $httpRequest->getLocale();
                        if ($preHandleResult->getLocale() != null) {
                            $locale = $preHandleResult->getLocale();
                        }
                        $redirectURL = $this->urlRouter->findURL(
                            $redirectControlerId,
                            $preHandleResult->getParameters(),
                            $preHandleResult->getMethodName(),
                            $locale
                        );
                    }

                    // do redirect
                    $this->log->info("redirect to " . $redirectURL);
                    header("Location: " . $redirectURL);
                    // terminate script execution
                    return;
                }
            }

            $controller = $this->container->get($routeInfo->getContrlollerId());

            $this->log->info("Retrieved controller " . get_class($controller) . " from container");

            // vyvolat metodu controlleru
            $methodName = $routeInfo->getMethodName();
            if ($methodName === null) { // $methodName not specified
                $methodName = "execute"; //  default to execute
            }

            if (!method_exists($controller, $methodName)) {
                $errorMessage = "Retrieved controller " . get_class($controller) . " doestn have requested method " . $methodName;
                $this->log->error($errorMessage);
                throw new Exception($errorMessage);
            }

            $calledMethodReflection = new ReflectionMethod($controller, $methodName);
            $methodParamDefs = $calledMethodReflection->getParameters();

            // TODO: udelat univerzalneji, aby se dalo pouzivat vice trid
            $methodParameters = [];
            foreach ($methodParamDefs as $paramDef) {
                $type = $paramDef->getType();
                if (!$type instanceof ReflectionNamedType) {
                    continue;
                }
                $reflClass = !$type->isBuiltin() ? new ReflectionClass($type->getName()) : null;
                if ($reflClass === null) {
                    throw new Exception("Nepodporovany typ parametru " . $type->getName());
                }
                if ($reflClass->getName() == HttpRequest::class || $reflClass->isSubclassOf(HttpRequest::class)) {
                    $methodParameters[] = $httpRequest;
                }
                if ($reflClass->getName() == Context::class || $reflClass->isSubclassOf(Context::class)) {
                    $methodParameters[] = $routeInfo->getContext();
                }
            }

            // call controller method
            /** @var ModelAndView $modelAndView return value of controller */
            $modelAndView = $calledMethodReflection->invokeArgs($controller, $methodParameters);

            // zavola postHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $interceptor->postHandle($httpRequest, $modelAndView);
            }

            if ($modelAndView != null && !$modelAndView->isRedirect()) { // a metoda neco vraci
                // vytvori view
                $view = $this->createAndPopulateView($routeInfo->getContext(), $modelAndView);

                // vyrendruje view
                $view->render();
            }

            if ($modelAndView != null // mame model and view
                && $modelAndView->isRedirect()) { // je redirect
                $redirectControlerId = $modelAndView->getViewName();

                if ($modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_URL) { // presmerovat na url
                    $redirectURL = $redirectControlerId;
                } elseif (
                    $modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_CONTROLLER
                ) { // presmerovat na controller
                    // najde url pro presmerovani
                    $locale = $httpRequest->getLocale();
                    if ($modelAndView->getLocale() != null) {
                        $locale = $modelAndView->getLocale();
                    }
                    $redirectURL = $this->urlRouter->findURL(
                        $redirectControlerId,
                        $modelAndView->getRedirectParameters(),
                        $modelAndView->getMethodName(),
                        $locale
                    );
                } else {
                    throw new Exception("Nepodporovany typ presmerovani " . $modelAndView->getRedirectType());
                }

                // do redirect
                $this->log->info("redirect to " . $redirectURL);

                if ($modelAndView->isPermanentRedirect()) { // is permanent redirect
                    header("HTTP/1.1 301 Moved Permanently");
                }
                header("Location: " . $redirectURL); // send redirect
                // Tell the browser nothing else to receive from this page.
                header("Content-Length: 0");
            }

            // run after completion of interceptors
            foreach (array_reverse($interceptors) as $interceptor) {
                $interceptor->afterCompletion($httpRequest, $modelAndView);
            }
        } catch (Throwable $exception) {
            // handle exception
            try {
                $result = false;
                if (!empty($httpRequest)) {
                    if (empty($httpRequest->getParameter(PageInfo::PAGE_INFO_INDENTIFIER))) {
                        $pageInfo = new PageInfo();
                        $pageInfo->setURL($httpRequest->getURL());
                        $httpRequest->addParameter(PageInfo::PAGE_INFO_INDENTIFIER, $pageInfo);
                    }
                    $pageInfo = $httpRequest->getParameter(PageInfo::PAGE_INFO_INDENTIFIER);

                    if ($routeInfo instanceof RouteInfo) {
                        $context = $routeInfo->getContext();
                    } else {
                        $context = new WebContext($httpRequest->getSession(), $pageInfo);
                        $context->setLocale($httpRequest->getLocale());
                        $context->setRouter($this->urlRouter);
                    }

                    $result = $this->handleExceptionByHandlers($exception, $httpRequest, $context);
                }
                if (!$result) {
                    //exception was not handled
                    $defaultHandler = new InternalServerErrorExceptionHandler($this->initErrorPage);
                    $defaultHandler->handleException($exception, null);
                }
            } catch (Throwable $exception) {
                $this->config->getAppLogger()->error('Handle of exception fail.', ['exception' => $exception]);

                if ($this->container->has("develserver") && $this->container->get("develserver")) {
                    throw $exception;
                }
            }

            if ($this->container->has("develserver") && $this->container->get("develserver")) {
                throw $exception;
            }

            die;
        }
    }

    /**
     * Zrapcovava http request do podoby zpracovavane dale
     *
     *
     * @return HttpRequest
     * @throws Exception
     */
    protected function populateHTTPRequest(ContainerInterface $container)
    {
        // filter_input ne vzdy funguje pro pouziti na  INPUT_SERVER
        //        $requestURL = filter_input(INPUT_SERVER, 'REQUEST_URI');
        $requestURL = $_SERVER['REQUEST_URI'];
        // najdi ? a odstran vse za nim, tj. parametry url
        $position = mb_strpos($requestURL, '?');
        if ($position !== false) {
            $requestURL = mb_substr($requestURL, 0, $position);
        }


        $locale = $this->localeResolver->resolveLocale($requestURL, $_REQUEST);
        $request = $this->localeResolver->modifyRequestURL($requestURL, $locale);

        $httpRequest = new HttpRequest($request, $_REQUEST, $locale);

        // parse body if exists
        if (array_key_exists("CONTENT_TYPE", $_SERVER)) {
            if (StringUtils::startsWith($_SERVER["CONTENT_TYPE"], 'application/json')) {
                $httpRequest->setParsedBody(json_decode(file_get_contents("php://input"), true));
            } elseif (StringUtils::startsWith($_SERVER["CONTENT_TYPE"], 'application/x-www-form-urlencoded')
                || StringUtils::startsWith($_SERVER["CONTENT_TYPE"], 'multipart/form-data')) {
                $httpRequest->setParsedBody($_POST);
            }
        }

        return $httpRequest;
    }

    /**
     * provede inicializaci aplikace
     */
    public function init()
    {
        try {
            $this->config->init();

            // inicializuje LocaleResolver
            // TODO: vyresit najak univerzalneji
            $this->localeResolver = $this->config->getLocaleResolver();
            // nastavuje viewResolvery pro vytvareni vire
            $this->viewResolvers = $this->config->getViewResolvers();
            // get exception handlers
            $this->exceptionHandlers = $this->config->getExceptionHandlers();

            // nastavuje logger
            $this->log = $this->config->getAppLogger();

            // nastavuje container
            $this->container = $this->config->getContainer();

            // pokud je definovan nejaky uzivarelsky session handler pouzij ho
            if ($this->container->has('session.sessionHandler')) {
                //                ini_set("session.gc_probability", 1);
                //                ini_set("session.gc_divisor", 1);
                $sessionHandler = $this->container->get('session.sessionHandler');
                session_set_save_handler($sessionHandler);
            }

            // inicializuje routery
            $this->urlRouter = $this->config->getRouter();
        } catch (Throwable $throwable) {
            if (
                $this->config->getContainer() !== null
            ) { // we have at least partially initialized container and thus logger
                $this->config->getAppLogger()->critical(
                    "Initialization of application faild.",
                    ['exception' => $throwable]
                );
            }

            //exception was not handled
            $defaultHandler = new InternalServerErrorExceptionHandler($this->initErrorPage);
            $defaultHandler->handleException($throwable, null);

            throw $throwable;
        }
    }

    protected function createAndPopulateView(Context $context, ModelAndView $modelAndView)
    {
        $this->log->info("Searching for view with name " . $modelAndView->getViewName());

        $view = null;
        foreach ($this->viewResolvers as $viewResolver) {
            $view = $viewResolver->buildView($modelAndView->getViewName()); // try to create view
            if ($view !== null) {
                break;
            }
        }

        if ($view == null) {
            throw new Exception("Nenalezeno potrebne view");
        }

        // provede inicializaci view
        $view->init($context, $modelAndView->getParameters());

        return $view;
    }

    /**
     * @param Throwable $exception
     * @param HttpRequest|null $httpRequest
     * @param Context|null $context
     * @return bool success of handle exception
     * @throws Exception handle exception fail
     */
    protected function handleExceptionByHandlers(Throwable $exception, ?HttpRequest $httpRequest, ?Context $context)
    {
        //go through handlers and find suitable handler
        foreach ($this->exceptionHandlers as $exceptionHandler) {
            if ($exceptionHandler->isFor($exception)) {
                $requestURL = $httpRequest !== null ? $httpRequest->getURL() : 'null';
                $this->log->log(
                    $exceptionHandler->getLogLevel(),
                    "Request to url $requestURL faild.",
                    ['exception' => $exception]
                );
                $modelAndView = $exceptionHandler->handleException($exception, $httpRequest);

                if ($modelAndView != null) {
                    if (!$modelAndView->isRedirect()) {
                        $view = $this->createAndPopulateView($context, $modelAndView);
                        $view->render();
                    } else {
                        throw new Exception('Handle exception can not be redirected');
                    }
                }
                return true;
            }
        }
        $this->log->error(
            "Not found exception handler on url :" . $httpRequest->getURL() . " ",
            ['exception' => $exception]
        );
        return false;
    }
}
