<?php


namespace IZON\MVC\Routers;


use FastRoute\RouteParser\Std;
use IZON\MVC\Context\WebContext;
use IZON\MVC\Exceptions\PageNotFoundException;
use IZON\MVC\HttpRequest;
use IZON\MVC\Interceptors\PageInfoInterceptor;
use IZON\MVC\PageInfo;
use IZON\MVC\Routers\Helpers\Route;
use IZON\Utils\Locale;
use League\Flysystem\Adapter\Local;
use League\Flysystem\Filesystem;
use Symfony\Component\Yaml\Yaml;


class YamlRouteDefinition implements RouteDefinition {

    /**
     * @var Std
     */
    protected $routeParser;

    /**
     * @var string path to files with routes
     */
    protected $routesFolderPath;

    /**
     * @var
     */
    protected $routes = [];

    /**
     * @var
     */
    protected $controllerRoutes = [];

    /**
     * @var
     */
    protected $interceptors = [];

    /**
     * @var array of arrays. Key is controller indentifier value is array of interceptor identifiers
     */
    protected $controllerInterceptors = [];

    /**
     * @var callable function to return name of routing table for provided locale
     */
    protected $localeRoutingTableNameCallback;

    /**
     * YamlRouteDefinition constructor.
     */
    public function __construct(string $routesFolderPath) {
        $this->routeParser = new Std();
        $this->routesFolderPath = $routesFolderPath;

        $this->localeRoutingTableNameCallback = function(Locale $locale) {
            $countryCode = mb_strtolower($locale->getCountry());
            return $countryCode;
        };

        // set interceptor that copies PageInfo to view
        $this->interceptors[] = new PageInfoInterceptor();
    }

    public function findRoute(HttpRequest $request) {
        $this->initRoutingTable($request->getLocale());
        $routingTableName = $this->getLocaleRoutingTableName($request->getLocale());

        if( !array_key_exists($routingTableName, $this->routes) ) {
            throw new \Exception("No routing table for locale '{$request->getLocale()->toLocaleTag()}'");
        }

        /** @var Route[] $routes */
        $routes = $this->routes[$routingTableName];

        $currentRoute = null;
        foreach($routes as $route) {
            list($matched, $params) = $route->match($request->getURL());
            if(!$matched) {
                continue;
            }
            $currentRoute = $route;
            break;
        }
        if($currentRoute === null) { // no route found
            return null;
        }

        foreach($params as $key => $value) {
            if(!is_numeric(
                $key
            )) { // numeric values represent index of match, we dont need them as we bind reqex values to names
                $request->addParameter($key, $value);
            }
        }
        $pageInfo = $currentRoute->getPageInfo();
        $pageInfo->setParameters($request->getParameters()); // set parameters
        $pageInfos = [];
        while(!empty($route->getParent())) {
            $route = $routes[$route->getParent()];
            $pageInfos[] = $route->getPageInfo();
        }
        $breadcumbs = array_reverse($pageInfos);
        $pageInfo->setBreadcrumbs($breadcumbs);
        $pageInfo->setURL($request->getURL());
        $request->addParameter(PageInfo::PAGE_INFO_INDENTIFIER, $pageInfo);

        $routeInfo = new RouteInfo();
        $routeInfo->setURL($request->getURL());
        $routeInfo->setContrlollerId($currentRoute->getControllerUID());
        if($currentRoute->getMethodName()) { // ma se posilat na jinou nez defaultni metodu
            $routeInfo->setMethodName($currentRoute->getMethodName());
            $pageInfo->setMethodName($currentRoute->getMethodName());
        }
        $routeInfo->setParameters($request->getParameters());

        // add context to page
        $context = new WebContext($request->getSession(), $pageInfo);
        $routeInfo->setContext($context);

        $interceptors = $this->interceptors;
        // add controller interceptors
        if( isset($this->controllerInterceptors[$currentRoute->getControllerUID()]) ) {
            $interceptors = array_merge($interceptors, $this->controllerInterceptors[$currentRoute->getControllerUID()]);
        }

        $routeInfo->setInterceptors($interceptors);

        // nastavuje id volaneho controlleru
        $request->setCalledControllerId($currentRoute->getControllerUID());

        return $routeInfo;
    }

    public function findURL($controllerId, array $parameters, $methodName, Locale $locale) {
        $this->initRoutingTable($locale);
        $routingTableName = $this->getLocaleRoutingTableName($locale);
        if( empty($methodName) ) {
            $methodName = '__NULL__';
        }
        // routes do not exist of $routingTableName, $controllerId and $methodName
        if( !is_array($this->controllerRoutes[$routingTableName][trim($controllerId)][$methodName]) ) {
            return null;
        }
        /** @var Route[] $routes */
        $routes = $this->controllerRoutes[$routingTableName][trim($controllerId)][$methodName];

        $paramNames = array_keys($parameters);
        foreach($routes as $route) {
            if( $route->hasAllVariables($paramNames) ) {
                return $route->generateURL($parameters);
            }
        }
        return null;
    }

    /**
     * initializes routing table of provided locale
     * @param Locale $locale
     */
    protected function initRoutingTable(Locale $locale) {
        $routingTableName = $this->getLocaleRoutingTableName($locale);
        if( !array_key_exists($routingTableName, $this->routes) ) {
            $localeRouteYamlFileName = $this->routesFolderPath .'/'. $routingTableName .'.yml';
            if( !file_exists($localeRouteYamlFileName) ) {
                throw new \Exception("File '". $localeRouteYamlFileName ."' with routes dows not exist.");
            }
            $localeRouteYaml = file_get_contents($localeRouteYamlFileName);
            $localeRoute = Yaml::parse($localeRouteYaml);

            $this->routes[$routingTableName] = array_map(
                function($routeParams) {
                    $routeParams['routeData'] = $this->routeParser->parse($routeParams['url']);
                    $route = new Route($routeParams);
                    return $route;
                },
                $localeRoute
            );

            $controllerIndex = [];
            foreach($this->routes[$routingTableName] as $route) {
                /** @var $route Route */
                if(!array_key_exists($route->getControllerUID(), $controllerIndex)) {
                    $controllerIndex[$route->getControllerUID()] = [];
                }
                // add array for method
                $methodName = $route->getMethodName() ? $route->getMethodName() : '__NULL__';
                if(!array_key_exists($methodName, $controllerIndex[$route->getControllerUID()])) {
                    $controllerIndex[$route->getControllerUID()][$methodName] = [];
                }
                // append route
                $controllerIndex[$route->getControllerUID()][$methodName][] = $route;
            }
            $this->controllerRoutes[$routingTableName] = $controllerIndex;
        }
    }

    /**
     * returns name for routing table for provided locale
     * @param Locale $locale
     * @return string
     */
    protected function getLocaleRoutingTableName(Locale $locale): string {
        $localeRoutingTableNameCallback = $this->localeRoutingTableNameCallback;
        $localeRoutingTableName = $localeRoutingTableNameCallback($locale);
        return $localeRoutingTableName;
    }

    /**
     * nastavuje interceptory platne pro vsechny controllery z teto route definition
     * @param array $interceptors
     */
    public function setInterceptors(array $interceptors) {
        $this->interceptors = $interceptors;
        // pridava PageInfoInterceptor ktery nacit informace o prave zobrazovane strance
        $this->interceptors[] = new PageInfoInterceptor();
    }

    /**
     * nastavuje interceptory jen pro specificke controllery
     * @param array $interceptors pole poli kde klicem je identifier controlleru a hodnota je pole interceptoru, ktere se na nem maji provezt
     */
    public function setControllerInterceptors(array $interceptors) {
        $this->controllerInterceptors = $interceptors;
    }

    /**
     * adds ides of interceptors to be called around $controllerId
     * @param string $controllerId
     * @param string[] $interceptorIdes
     * @throws Exception
     */
    public function addControllerInterceptors($controllerId, array $interceptorIdes) {
        if(array_key_exists($controllerId, $this->controllerInterceptors)) {
            throw new Exception("Interceptors for controller $controllerId already added.");
        }
        $this->controllerInterceptors[$controllerId] = $interceptorIdes;
    }

}
