<?php

namespace IZON\MVC\Views\Resolvers;

use Exception;

use IZON\Logs\Logger;

use \IZON\MVC\Views\Resolvers\ViewResolver;
use \IZON\MVC\Views\PHPView;

/**
 * resolves what view to use for rendering of admin
 * all admin views must start with admin/ prefix to be processed by this view resolver
 * 
 * @author IZON s.r.o. <info@izon.cz>
 * @package IZON\MVC\Views
 */
class PHPViewResolver implements ViewResolver {
    
    /**
     * path relative each moduleDir
     */
    const DEFAULT_LIBRARIES_DIR = "_libs";
    
    /**
     * @var string directory to search for views
     */
    protected $viewsDir;
    
    /**
     * @var string[] dirs containing libraries for use in view
     */
    protected $librariesDirs;

    /**
     * @var string[] files containing libraries used for rendering the view
     */
    protected $librariesFiles = [];
    
    /**
     * @var array array of patterns for vews that require output buffer to be disabled
     */
    protected $dissableOutputBufferPatterns = [];
    
    /**
     * @var array key is http path to prefix files value is fs directory to search file for
     */
    protected $staticContentDirs;
    
    /**
     * @var string[] prefiexs to cicle through for statis files to balance load for fifferent servers
     */
    protected $staticContentPrefixes = [""];
    
    /**
     * @var Logger 
     */
    protected $log;
    
    
    function __construct($viewsDir, $staticContentDirs, $config = []) {
        $this->viewsDir = $viewsDir;
        
        // redefine libraries dirs
        if(array_key_exists("librariesDirs", $config) ) {
            $this->librariesDirs = $config["librariesDirs"];
        } else {
            $this->librariesDirs = [$this->viewsDir .'/'. self::DEFAULT_LIBRARIES_DIR];
        }
        
        // redefine libraties files
        if( array_key_exists("librariesFiles", $config) ) {
            $this->librariesFiles = $config["librariesFiles"];
        }
        
        // dissableOutputBufferPatterns
        if( array_key_exists('dissableOutputBufferPatterns', $config) ) {
            $this->dissableOutputBufferPatterns = $config['dissableOutputBufferPatterns'];
        }
        
        // set directories for static content
        $this->staticContentDirs = $staticContentDirs;
        
        // redefine prefiexes for static content
        if( array_key_exists("staticContentPrefixes", $config) ) {
            $this->staticContentPrefixes = $config["staticContentPrefixes"];
        }
        
        $this->log = Logger::getLogger(self::class);
    }

    
    public function buildView($viewName) {
        $this->log->info("Trying to resolve view $viewName");
        
        $viewTemplatePath = $this->viewsDir ."/". $viewName . PHPView::VIEW_SUFFIX;
        // nepouziva view
        if( !file_exists($viewTemplatePath) ) {
            return NULL;
        }
        
        $view = new PHPView($viewTemplatePath);
        $view->setLibrariesDirs($this->librariesDirs);
        $view->setLibrariesFiles($this->librariesFiles);
        $view->setStaticContentDirs($this->staticContentDirs);
        $view->setStaticContentPrefixes($this->staticContentPrefixes);
        
        if( $this->disableOutputBufferForView($viewName) ) {
            $view->setUseOutputBuffer(false);
        }
        
        return $view;
    }
    
    /**
     * returns info if to disable output buffer for view
     * @param string $viewName
     * @return bool
     */
    public function disableOutputBufferForView(string $viewName): bool {
        foreach($this->dissableOutputBufferPatterns as $pattern) {
            if( fnmatch($pattern, $viewName) ) {
                return true;
            }    
        }
        return false;
    }
}
