<?php

namespace IZON\MVC;

use IZON\Utils\Locale;

/**
 * trida pouzivana k predavani informaci,
 * ktere view se ma pouzit a kat potrebnych pro toto view
 */
class ModelAndView {
    
    const REDIRECT_TYPE_NONE = 'none';
    
    const REDIRECT_TYPE_CONTROLLER = 'controller';
    
    const REDIRECT_TYPE_URL = 'url';
    
    /**
     * nazev view, ktere se ma pouzit
     * @var string 
     */
    private $viewName;
    
    /**
     * jaky typ pesmerovani se ma pouzit
     * moznosti: 
     * none - neprovadet presmerovani, vyrenderovat view s danym jmenem
     * controller - udelat presmerovani na controller se zadanym identifierem 
     * url - udelat presmerovani na zadane url
     * @var string
     */
    private $redirectType = 'none';
    
    /**
     * parametry predavane jen pri redirectu, ktere se nastavi do adresy
     * @var array 
     */
    private $redirectParameters;
    
    /**
     * @var string jaka metoda se ma zavolat
     */
    protected $methodName = NULL;


    /**
     * @var boolean jestli se ma provest pernamentni presmerovani
     */
    private $permanentRedirect = false;

    /**
     * parametry predavane do view, pokud je redirect
     * @var array 
     */
    private $parameters = NULL;

    /**
     * @var Locale na jake locale se ma presmerovat  
     */
    protected $locale = NULL;


    /**
     * 
     * @param string $viewName bud jake view se ma pouzit nebo controller na ktery se ma presmerovat 
     * @param array $parameters pole parametru
     * @param boolean $redirect jestli je redirect
     */
    function __construct($viewName, 
                            $parameters = [], 
                            $redirect = false ) {
        $this->viewName   = $viewName;
        $this->parameters = $parameters;
        
        if( $redirect == true ) {
            $this->redirectType = self::REDIRECT_TYPE_CONTROLLER;
            $this->redirectParameters = $parameters;
        }
    }

    function getViewName() {
        return $this->viewName;
    }

    function setViewName($viewName) {
        $this->viewName = $viewName;
    }
    
    public function isRedirect() {
        return $this->redirectType != 'none';
    }
    
//    public function setRedirect($redirect) {
//        $this->redirect = $redirect;
//    }
    
    function putParameter($paramName, $param) {
        $this->parameters[$paramName] = $param;
    }

    /**
     * vrati pole se vsemi parametry
     */
    function getParameters() {
        return $this->parameters;
    }

    /**
     * vrati konkretni parametr
     */
    function getParameter($parameter) {
        if(trim($parameter) != '') {
            return $this->parameters[$parameter];
        }
    }
    
    function getRedirectType() {
        return $this->redirectType;
    }

    function getRedirectParameters() {
        return $this->redirectParameters;
    }
    
    function getMethodName() {
        return $this->methodName;
    }

    function isPermanentRedirect() {
        return $this->permanentRedirect;
    }
    
    function getLocale() {
        return $this->locale;
    }
    
    /**
     * vytvori presmerovanu na zadany controller
     * @param string $controllerId
     * @param array $parameters parametry, se kterymi chceme provezt presmerovani, parametry pridane v nejakem interceptoru se nepridavani do vygenerovane adresy
     * @param string $methodName na jakou metodu controlleru se ma pozadavek poslat
     * @param boolean $permanentRedirect jestli ma byt permanentni presmrovani (http 301 Moved Permanently)
     * @param Locale $locale na jake locale se ma presmerovat
     * @return ModelAndView
     */
    public static function createControllerRedirect($controllerId, $parameters = [], $methodName = NULL, $permanentRedirect = false, Locale $locale = NULL) {
        $modelAndView = new ModelAndView($controllerId, NULL);
        $modelAndView->redirectParameters = $parameters;
        $modelAndView->redirectType = self::REDIRECT_TYPE_CONTROLLER;
        $modelAndView->permanentRedirect = $permanentRedirect;
        $modelAndView->methodName = $methodName;
        $modelAndView->locale = $locale;
        return $modelAndView;
    }
    
    /**
     * vytvori presmerovanu na zadane url
     * @param string $url
     * @return ModelAndView Description
     */
    public static function createURLRedirect($url, $permanentRedirect = false) {
        $modelAndView = new ModelAndView($url, NULL);
        $modelAndView->redirectType = self::REDIRECT_TYPE_URL;
        $modelAndView->permanentRedirect = $permanentRedirect;
        return $modelAndView;
    }
}