<?php

namespace IZON\MVC\Routers;

use \Exception;

use \IZON\Utils\Locale;
use IZON\MVC\Locale\LocaleResolver;

use IZON\MVC\HttpRequest;
use IZON\MVC\Routers\RouteInfo;
use \IZON\MVC\Exceptions\PageNotFoundException;

/**
 * rozhrani, ktere podle url najde controller, ktery se ma zavolat
 */
class Router {
    
    /**
     * defninice routovani, prochazeji se postupne dokud nejaka nevrati schodu
     * @var RouteDefinition 
     */
    protected $urlRouteDefinitions = [];
    
    /**
     * interceptory, ktere se maji vykonat pri kazdem requestu
     * @var array 
     */
    protected $commonInterceptors = [];
    
    /**
     * @var LocaleResolver 
     */
    protected $localeResolver = NULL;

    /**
     * @var boolean pokud je true tak pro stranky s koncovym lomitkem v url provede externi presmervani na stranku bez neho pokud takove stranky existuji
     */
    protected $redirectToURLWithoutTailingSlash = false;
    
    
    public function __construct(array $urlRouteDefinitions) {
        $this->urlRouteDefinitions = $urlRouteDefinitions;
    }
    
    /**
     * pro request najde informace, jaky controller se ma volat a jake interceptory
     * @param HttpRequest $httpRequest
     * @return RouteInfo
     * @throws Exception
     */
    public function findRoute(HttpRequest $httpRequest) {
        $url = $httpRequest->getURL();
        // removes tailing /
        if( $this->redirectToURLWithoutTailingSlash
            && \IZON\String\endsWith($url, "/")
            &&  $url != '/') {
            $urlWithoutSlash = mb_substr($url, 0, mb_strlen($url)-1); // urizni koncove lomitko
            $newRequest = new HttpRequest($urlWithoutSlash, $httpRequest->getParameters(), $httpRequest->getLocale());
            
            // try to find route for url without tailing /
            foreach( $this->urlRouteDefinitions as $definition ) {
                $routeInfo = $definition->findRoute($newRequest);
                if( $routeInfo != NULL ) {
                    // add locale data to url
                    $newUrl = $this->localeResolver->modifyOutpultURL($urlWithoutSlash, $newRequest->getLocale());
                    $routeInfo = new RouteInfo();
                    $routeInfo->setRedirect(true);
                    $routeInfo->setUrl($newUrl);
                    
                    // send redirect to new address
                    return $routeInfo;
                }
            }
        }
        
        // try to find address as is
        foreach( $this->urlRouteDefinitions as $definition ) {
            $routeInfo = $definition->findRoute($httpRequest);
            if( $routeInfo != NULL ) {
                // prida spolecne interceptory na zacatek pole
                $interceptors = $this->commonInterceptors;
                $interceptors = array_merge($interceptors, $routeInfo->getInterceptors());
                $routeInfo->setInterceptors($interceptors);
                
                // prepare context
                $context = $routeInfo->getContext();
                if( $context !== NULL ) {
                    $context->setLocale($httpRequest->getLocale());
                    $context->setRouter($this);
                }
                
                return $routeInfo;
            }
        }
        
        // no routing find
        throw new PageNotFoundException("Routing of address ". $httpRequest->getUrl() ." with parameters: ". print_r($httpRequest->getParameters(), ture) ." not found");
    }
    
    /**
     * pro controller s id $controllerId najde adresu pres kterou je ho mozno volat
     * @param string $controllerId id controlleru
     * @param array $parameters parametry, ptere se maji predat do adresy
     * @param string $methodName nazev metody, ktera se ma na controlleru zavolat
     * @param string|Locale $locale
     * @return string
     * @throws Exception
     */
    public function findURL($controllerId, array $parameters, $methodName, $locale) {
        foreach( $this->urlRouteDefinitions as $definition ) {
            $url = $definition->findURL($controllerId, $parameters, $methodName, $locale);
            if( $url != NULL ) {
                $url = $this->localeResolver->modifyOutpultURL($url, $locale);
                return $url;
            }
        }
        throw new Exception("URL for controller $controllerId and parameters ". print_r($parameters, ture) ." not found");
    }
    
    function setCommonInterceptors(array $commonInterceptors) {
        $this->commonInterceptors = $commonInterceptors;
    }

    function setLocaleResolver(LocaleResolver $localeResolver) {
        $this->localeResolver = $localeResolver;
    }
    
    function setRedirectToURLWithoutTailingSlash($redirectToURLWithoutTailingSlash) {
        $this->redirectToURLWithoutTailingSlash = $redirectToURLWithoutTailingSlash;
    }
}