<?php

namespace IZON\MVC\Locale;

use \IZON\Utils\Locale;

/**
 *  Locale resolver that takes locale from start of url
 */
class URLLocaleResolver implements LocaleResolver {
    /**
     *
     * @var string defaultni locale, ktere se ma pouzit pokud neni zjisteno jine 
     */
    protected $defaultLocaleURLPrefix;
    /**
     *
     * @var array podporovane locale pro stranku, pokud nalezeno jine nez podporovane preklopi se defaultni 
     */
    protected $supportedLocales;
    /**
     *
     * @var string v jakem adresari se pocale nachazeji 
     */
    protected $localeDir;
    /**
     *
     * @var string 
     */
    protected $tmpLocaleDir;

    /**
     * @param string[] $supportedLocaleTags
     * @param string $defaultLocaleURLPrefix
     * @param string $localeDir
     * @param string $tmpLocaleDir
     */
    function __construct($supportedLocaleTags, $defaultLocaleURLPrefix, $localeDir, $tmpLocaleDir) {
        $this->defaultLocaleURLPrefix = $defaultLocaleURLPrefix;
        
        $this->supportedLocales = [];
        foreach($supportedLocaleTags as $languageURLPrefix => $supportedLocaleTag) {
            $this->supportedLocales[$languageURLPrefix] = Locale::forLocaleTag($supportedLocaleTag);
        }
        
        $this->localeDir = $localeDir;
        $this->tmpLocaleDir = $tmpLocaleDir;
    }

    /**
     * 
     * @param string $requestURL
     * @param array $requestParams
     * @return \IZON\MVC\Locale
     */
    public function resolveLocale($requestURL, $requestParams) {
        $selectedLocale = $this->supportedLocales[$this->defaultLocaleURLPrefix];
        $domain = $selectedLocale->getLanguage();
        foreach($this->supportedLocales as $key => $locale) {
            if( $key != $this->defaultLocaleURLPrefix // neni defaultni locale
                    && (\IZON\String\startsWith($requestURL, '/'. $key .'/') 
                        || $requestURL == '/'. $key ) ) {
                $selectedLocale = $locale;
                $domain = $selectedLocale->getLanguage();
                break;
            }
        }

        // create locale string as used in OS
        $osLocaleTag = str_replace("-", '_', $selectedLocale->toLocaleTag());
        $newDomain = $domain;

        // tries to set exactlly $selectedLocale locale, if not found on os select one of 'cs_CZ', 'cs', 'en_US', 'en'
        $osFallbackLocaleTag = setlocale(LC_ALL, $osLocaleTag, 'cs_CZ', 'cs', 'en_US', 'en');
        // set formating for numbers to english NEEDS TO BE TESTED
        setlocale(LC_NUMERIC, 'C');
        putenv('LANGUAGE='. $osFallbackLocaleTag);
        
        // location of translation .mo file
        $localeFilePath = $this->localeDir . '/' . $osLocaleTag . "/LC_MESSAGES/$domain.mo";
        if( file_exists($localeFilePath) ) { // we have locale 
            $mtime = filemtime($localeFilePath);

            // where to store locale
            $newLocaleDir = $this->tmpLocaleDir .'/'. $osFallbackLocaleTag . '/LC_MESSAGES';
            $newDomain = 'messages_' . $domain;
            
            $timestampedDomain = $newDomain .'_'. $mtime;
            $newLocaleFilePath = $newLocaleDir . '/' . $timestampedDomain . '.mo';

            if(!is_dir($newLocaleDir)) {
                mkdir($newLocaleDir, 0777, true);
            }
            if(!file_exists($newLocaleFilePath)) {
                copy($localeFilePath, $newLocaleFilePath);
                chmod($newLocaleFilePath, 0777);
            }
            
            bindtextdomain($timestampedDomain, $this->tmpLocaleDir);
            textdomain($timestampedDomain);
            bind_textdomain_codeset($timestampedDomain, 'UTF-8');
        }

        return $selectedLocale;
    }

    /**
     * 
     * @param string $requestURL
     * @param Locale $locale
     */
    public function modifyRequestURL($requestURL, Locale $locale) {
        // odstran language z url
        $localeName = array_search($locale, $this->supportedLocales);
        if( $localeName === false ) {
            throw new Exception("Unsupported locale ". $locale->toLocaleTag());
        }
        
        if( $localeName == $this->defaultLocaleURLPrefix ) { // is default locale do not modify
            return $requestURL;
        }
        
        if($requestURL == "/" . $localeName 
            || \IZON\String\startsWith($requestURL, "/" . $localeName . "/")) { // uriznout jen pokud je samotne nebo zacina a ma dalsi cast
            $ret = mb_substr($requestURL, mb_strlen("/" . $localeName));
            if($ret == '') {
                $ret = '/';
            }
            return $ret;
        } else {
            return $requestURL;
        }
    }

    /**
     * adds url prefix that diferentiates locales
     * @param string $requestURL
     * @param Locale $locale
     * @return string url s pridanymi informcemi o locale
     */
    public function modifyOutpultURL($requestURL, Locale $locale) {
        // odstran language z url
        $localeName = array_search($locale, $this->supportedLocales);
        if( $localeName === false ) {
            throw new Exception("Unsupported locale ". $locale->toLocaleTag());
        }
        
        if( $localeName == $this->defaultLocaleURLPrefix ) { // is default locale do not modify
            return $requestURL;
        }
        
        return "/". $localeName . $requestURL;
    }

}