<?php

namespace IZON\MVC\View\Resolvers;

use Exception;

use IZON\Logs\Logger;

use \IZON\MVC\View\Resolvers\ViewResolver;
use \IZON\MVC\View\PHPLayoutView;

/**
 * trida renderujici view, resici cachovani rendrovani view, skladajici view 
 * z jednotlivych casti, ...
 * jednotliva view jsou ulozena v adresarich
 * pri devel modu naincluduje nekde dole ve strance misto na debug informace
 *
 * @author David Dryml <dryml@izon.cz>
 * @copyright Copyright 2009, HI Software s.r.o.
 *
 * @version v1.0 rc
 *
 * @package MVC
 */
class PHPLayoutViewResolver implements ViewResolver {
 
    /**
     * @var string directory containing views
     */
    protected $viewsDir = __BASE_DIR__ ."/views";
    
    /**
     * @var string containing layouts 
     */
    protected $layoutsDir = __BASE_DIR__ ."/layouts";
    
    /**
     * @var string dir containing libraries for use in view 
     */
    protected $librariesDir = NULL;

    /**
     * @var string files containing libraries used for rendering the view
     */
    protected $librariesFiles = [];
    
    /**
     * konfigurace nacitanych view
     * @var array 
     */
    protected $viewsConfig = [];
    
    /**
     * @var array) 
     */
    protected $staticContentConfig = [];
    
    /**
     * @var Logger 
     */
    protected $log = NULL;
    
    
    function __construct(array $viewsConfig, array $config = []) {
        $this->log = Logger::getLogger(self::class);
        
        if( !isset($viewsConfig["_default"]["layout"]) ) {
            throw new Exception('V poli musi byt nastaven defaultni layout (\$viewsConfig["_default"]["layout"])');
        }
        $this->viewsConfig = $viewsConfig;
        
        if( isset($config["librariesDir"] ) ) {
            $this->librariesDir = $config["librariesDir"];
        }
        
        if( isset($config["staticContent"]) ) {
            $this->staticContentConfig = $config["staticContent"];
        }
    }

    public function buildView($viewName) {
        $this->log->info("Trying to resolve view $viewName");
        $layoutFileName = "";
        foreach($this->viewsConfig as $configWiewName => $config) { // nazev view zacina
            if( fnmatch($configWiewName, $viewName) ) {
                $layoutFileName = $config["layout"];
                break;
            }
        }
        if( $layoutFileName == NULL ) { // neni zmeneno nastavit defaultni
            $layoutFileName = $this->viewsConfig["_default"]["layout"];
        }
        
        $this->log->info("Using template $layoutFileName");
        
        // TODO: umoznit pouzivat i jina view
        $view = new PHPLayoutView($layoutFileName, 
                                  $this->layoutsDir,
                                  $viewName,
                                  $this->viewsDir,
                                  $this->librariesDir,
                                  $this->librariesFiles,
                                  $this->staticContentConfig);
        
        return $view;
    }
    
    /**
     * returns info which files to minify
     * @return array
     */
    public function getStaticContentToMinify() {
        $sourceDir = $this->staticContentConfig["staticContentDir"];
        if( !file_exists($sourceDir)
           || !is_dir($sourceDir) ) {
            return [];
        }
        $targetDir = $this->staticContentConfig["minifiedContentDir"];
        if( !file_exists($targetDir)
           || !is_dir($targetDir) ) {
            return [];
        }
        
        $jsDirsToMinify = $this->staticContentConfig["jsDirsToMinify"];
        $contentToMinify = [];
        foreach($jsDirsToMinify as $jsSourceDir) {
            $jsFullSourceDir = $sourceDir ."/". $jsSourceDir;
            if( file_exists($jsFullSourceDir)
                && is_dir($jsFullSourceDir) ) {
                $contentToMinify[] = ["sourceDir" => $jsFullSourceDir, "targetDir" => $targetDir ."/". $jsSourceDir, "type" => "js"];
            }
        }
        
        return $contentToMinify;
    }
}
