<?php

namespace IZON\MVC\View;

use \IZON\MVC\Routers\Router;
use \IZON\MVC\ModelAndView;

use IZON\Utils\Locale;

/**
 * trida renderujici view, resici cachovani rendrovani view, skladajici view z jednotlivych casti, ...
 * jednotliva view jsou ulozena v adresarich
 * pri devel modu naincluduje nekde dole ve strance misto na debug informace
 */
class PHPView implements View {
    
    const VIEW_SUFFIX = ".tmpl.php";

    /**
     * @var string name of view to render
     */
    protected $viewName;
    
    /**
     * @var Locale v jakym locale se ma view zjistit
     */
    protected $locale;
    
    /**
     * @var array parameters to use in template
     */
    protected $parameters;

    /**
     * @var Router router potrebny pro spetny preklad adresy
     */
    protected $router;
    
    /**
     * @var string directory, where to find views 
     */
    protected $viewsDir;

    /**
     * @var string|null dir containing libraries for use in view 
     */
    protected $librariesDir = NULL;

    /**
     * @var string files containing libraries used for rendering the view
     */
    protected $librariesFiles = [];
    
    /**
     * @var boolean if true and stratic file does't exist throw exeption
     */
    protected $strictStaticContentInclude = false;
    
    /**
     * @var boolean if true prefare to use minified static content 
     */
    protected $prefareMinifiedStacitContent = false;

    /**
     * @var string directory where to find static content
     */
    protected $staticContentDir = null;
    
    /**
     * @var array prefixes to use for non minified static content
     */
    protected $staticContentURLPefixes = [];
    
    /**
     * @var string directory where to find minified static content
     */
    protected $minifiedContentDir = null;
    
    /**
     * @var array prefixes to use for minified static content
     */
    protected $minifiedContentURLPefixes = [];
    
    /**
     * @var integer 
     */
    protected $lastContentPrefixIndex = 0;


    /**
     * view to render
     * @var View 
     */
    private static $applicationView;
    
    /**
     * vytvari view podle zadaneho jmena a predava mu parametry
     */
    function __construct($viewName, $viewsDir = __BASE_DIR__ ."/views", $librariesDir = NULL, $librariesFiles = [], $staticContentConfig = []) {
    	$this->viewName = $viewName;
        
        $this->viewsDir = $viewsDir;
        
        $this->librariesDir = $librariesDir;
        $this->librariesFiles = $librariesFiles;
        
        if( isset($staticContentConfig["staticContentDir"])
            && isset($staticContentConfig["staticContentURLPefixes"]) ) {
            $this->staticContentDir = $staticContentConfig["staticContentDir"];
            $this->staticContentURLPefixes = $staticContentConfig["staticContentURLPefixes"];
        } else {
            // fallback na
            $this->staticContentDir = __BASE_DIR__ ."/www";
            $this->staticContentURLPefixes = [""];
        }
        
        if( isset($staticContentConfig["minifiedContentDir"])
            && isset($staticContentConfig["minifiedContentURLPefixes"]) ) {
            $this->minifiedContentDir = $staticContentConfig["minifiedContentDir"];
            $this->minifiedContentURLPefixes = $staticContentConfig["minifiedContentURLPefixes"];
        }
        if( isset($staticContentConfig["prefareMinified"]) ) {
            $this->prefareMinifiedStacitContent = $staticContentConfig["prefareMinified"] 
                && $this->minifiedContentDir !== null;
        }
    }
    
    function getLocale() {
        return $this->locale;
    }
    
    function getParameters() {
    	return $this->parameters;
    }
    
    function getParameter($paramName) {
        if( isset($this->parameters[$paramName]) ) {
            return $this->parameters[$paramName];
        } else {
            return NULL;
        }
    }
    
    function hasParameter($paramName) {
        return array_key_exists($paramName, $this->parameters);
    }
    
    function getLang() {
        return $this->getLocale()->getLanguage();
    }
    
    /**
     * vrati router
     */
    function getRouter() {
        return $this->router;
    }
    
    /**
     * provede vykresleni celeho view
     */
    function render() {
        // load view libraries
        if( $this->librariesDir != NULL ) { // some library dir is set
            foreach(scandir($this->librariesDir) as $file) {
                if( \IZON\String\endsWith($file, ".php") ) {
                    require_once $this->librariesDir ."/". $file;
                }
            }
        }
        
    	// nacte pozadovanou sablonu
    	// TODO pridavat $this->getLang() pred priponu pro podporu vice jazyku
    	require_once $this->viewsDir  ."/". $this->viewName . self::VIEW_SUFFIX;
    }
    
    /**
     * nacte staticke aplikacni view
     * @return PHPView aplikacni viw
     */
    static function getApplicationView() {
    	return self::$applicationView;
    }
    
    /**
     * ulozi staticke aplikacni view
     */
    static function setApplicationView($applicationView) {
    	self::$applicationView = $applicationView; 
    }

    public function init(Router $router, 
                         ModelAndView $modelAndView,
                         Locale $locale) {
        $this->router = $router;
        $this->parameters = $modelAndView->getParameters();
        $this->locale = $locale;
        
        self::setApplicationView($this);
    }
    
    
    public function getStaticFileURL($relativeLocaltion) {
        if( $this->prefareMinifiedStacitContent ) {
            $fileLocaltion = $this->minifiedContentDir ."/". $relativeLocaltion;
            if( file_exists($fileLocaltion)
                && !is_dir($fileLocaltion) )  {
                return $this->getNextMinifiedStaticContentURLPrefix() ."/". $relativeLocaltion ."?". filemtime($fileLocaltion);
            }
        }
        
        $fileLocaltion = $this->staticContentDir ."/". $relativeLocaltion;
        if( file_exists($fileLocaltion)
            && !is_dir($fileLocaltion) )  {
            return $this->getNextStaticContentURLPrefix() ."/". $relativeLocaltion ."?". filemtime($fileLocaltion);
        }
        
        if( $this->strictStaticContentInclude ) {
            throw new Exception("Static file ". $relativeLocaltion ." not found.");
        } else {
            return ;
        }
    }
    
    protected function getNextMinifiedStaticContentURLPrefix() {
        $this->lastContentPrefixIndex = ($this->lastContentPrefixIndex + 1) % count($this->minifiedContentURLPefixes);
        $prefix = $this->minifiedContentURLPefixes[$this->lastContentPrefixIndex];
        return $prefix;
    }
    
    protected function getNextStaticContentURLPrefix() {
        $this->lastContentPrefixIndex = ($this->lastContentPrefixIndex + 1) % count($this->staticContentURLPefixes);
        $prefix = $this->staticContentURLPefixes[$this->lastContentPrefixIndex];
        return $prefix;
    }
}