<?php

namespace IZON\MVC;

use Exception;
use IZON\MVC\Exceptions\PageNotFoundException;
use Throwable;
use IZON\DI\Container;
use IZON\Logs\Logger;
use IZON\MVC\Context\Context;
use IZON\MVC\Exceptions\Handlers\AppExceptionHandler;
use IZON\MVC\Exceptions\InternalServerErrorException;
use IZON\MVC\Interceptors\PreHandleResult;
use IZON\MVC\Locale\LocaleResolver;
use IZON\MVC\Routers\Router;
use IZON\Utils\Locale;
use ReflectionMethod;
use ReflectionParameter;

/**
 * aplikace
 */
class App {

    /**
     * v jakem adresari se nachazi cela aplikace
     * @var string
     */
    protected $baseDir = NULL;

    /**
     * jaka trida se ma pouzit pro nacitani konfiguraci
     * @var class-string class name
     */
    protected $configClass = NULL;

    /**
     * konfugurace aplikace
     * @var Config
     */
    protected $config = NULL;

    /**
     * logger pro logovani prubeho vykonavani aplikace
     * @var Logger $log
     */
    protected $log = NULL;

    /**
     * DI container z ktereho je mozne vytahovat controllery
     * @var Container
     */
    protected $container = NULL;

    /**
     * router, ktery podle url vrati ktery controller se ma zavoloat
     * @var Router
     */
    protected $urlRouter;

    /**
     * zjisti jake locale se ma pouzit
     * @var LocaleResolver
     */
    protected $localeResolver;

    /**
     * slouzi k ziskani view, ktere se ma vyrenderovat
     * @var array
     */
    protected $viewResolvers;

    /**
     * resolvery pro zpracovani vyjimek vyhozenych systemech
     * @var AppExceptionHandler[]
     */
    protected $exceptionHandlers = [];
    /**
     * relative Path to file that require on init app error
     * @var string
     */
    protected $initErrorPage = 'errors/500.php';

    /**
     * App constructor.
     * @param $baseDir
     * @param string $configClass
     * @param array $parameters array of optional parameters
     *          supported array:
     *          ['initErrorPage' => 'relative path to error file '] - error in init the application
     */
    public function __construct(
        $baseDir,
        $configClass = Config::class,
        $parameters = []
    ) {

        // sets basedir of aplication for easier url handling
        // TODO: potom nabradir necim co se jednoduseji spravuje
        // FIXME: ve verzi do nedrzi spetnou kompatibilitu odstranit
        define("__BASE_DIR__", $baseDir);

        $this->baseDir = $baseDir;
        $this->configClass = $configClass;

        if ($parameters['initErrorPage']) {
            $this->initErrorPage = $parameters['initErrorPage'];
        }
    }

    /**
     * provede inicializaci aplikace
     */
    public function init() {
        try {
            $configClass = $this->configClass;
            // vytvori konfiguraci
            /** @var Config $config */
            $config = new $configClass($this->baseDir);
            $config->init();
            $this->config = $config;

            /// set all properties necessary for error handling
            // nastavuje container
            $this->container = $config->getContainer();
            // nastavuje logger
            $this->log = $config->getAppLogger();
            // get exception handlers
            $this->exceptionHandlers = $this->config->getExceptionHandlers();

            // inicializuje LocaleResolver
            // TODO: vyresit najak univerzalneji
            $this->localeResolver = $this->config->getLocaleResolver();
            // nastavuje viewResolvery pro vytvareni vire
            $this->viewResolvers = $this->config->getViewResolvers();

            // pokud je definovan nejaky uzivarelsky session handler pouzij ho
            if ($this->container->has('session.sessionHandler')) {
                //                ini_set("session.gc_probability", 1);
                //                ini_set("session.gc_divisor", 1);
                $sessionHandler = $this->container->get('session.sessionHandler');
                session_set_save_handler($sessionHandler);
            }

            // inicializuje routery
            $this->initRouter($config);

        } catch (Exception $exception) {
            $this->handleExceptionByHandlers($exception, null);
        } catch (Throwable $throwable) {
            $this->handleExceptionByHandlers($throwable, null);
        }
    }

    /**
     * spusti zpracocani dotazu db dotazu
     */
    public function dispatch() {
        // vytvori a nastavi request
        $httpRequest = $this->populateHTTPRequest();
        try {
            // vytvori session id a posle ho prohlizeci, potom session odemce
            $httpRequest->getSession()->init();

            // TODO: asi vydejit obsluhu presmerovani nekam ma jedno misto
            // najde routovani
            $routeInfo = $this->urlRouter->findRoute($httpRequest);
            if ($routeInfo->getRedirect() != 0) {
                // do redirect
                $this->log->info("redirect from router to " . $routeInfo->getUrl());
                header("Location: " . $routeInfo->getUrl());
                // ukoncit vykonavani
                return;
            }

            $this->log->info("Routing url " . $httpRequest->getUrl() . " to " . $routeInfo->getContrlollerId());

            // najde interceptory
            $interceptors = $routeInfo->getInterceptors();
            // if interceptor specified by identifier, replace by interceptor from container
            for ($i = 0; $i < count($interceptors); $i++) {
                $interceptor = $interceptors[$i];
                if (is_string($interceptor)) {
                    $this->log->info("Reatriving interceptors " . $interceptor . " from container");
                    $interceptors[$i] = $this->container->get($interceptor);
                }
            }

            $this->log->info("Retrieved interceptors for controller " . $routeInfo->getContrlollerId() . " from container");

            // zavola preHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $this->log->info("Calling preHandle of interceptor " . get_class($interceptor) . "");
                $preHandleResult = $interceptor->preHandle($httpRequest);

                if ($preHandleResult instanceof PreHandleResult) { // ma se provect presmerovani
                    // TODO: sestavit presmerovani
                    $redirectURL = $preHandleResult->getRedirectAddress();
                    if ($preHandleResult->isInternalRedirect()) {
                        $redirectControllerId = $preHandleResult->getControllerId();
                        // najde url pro presmerovani
                        $methodName = $preHandleResult->getMethodName();
                        $locale = $httpRequest->getLocale();
                        if ($preHandleResult->getLocale() != NULL) {
                            $locale = $preHandleResult->getLocale();
                        }
                        $redirectURL = $this->urlRouter->findURL($redirectControllerId, $preHandleResult->getParameters(), $preHandleResult->getMethodName(), $locale);
                    }

                    // do redirect
                    $this->log->info("redirect to " . $redirectURL);
                    header("Location: " . $redirectURL);

                    // ukoncit vykonavani
                    return;

                    // TODO:
                }
            }

            $controller = $this->container->get($routeInfo->getContrlollerId());

            $this->log->info("Retrieved controller " . get_class($controller) . " from container");

            // vyvolat metodu controlleru
            $methodName = $routeInfo->getMethodName();
            if ($methodName === NULL) { // $methodName not specified
                $methodName = "execute"; //  default to execute
            }

            if (!method_exists($controller, $methodName)) {
                $errorMessage = "Retrieved controller " . get_class($controller) . " doestn have requested method " . $methodName;
                $this->log->error($errorMessage);
                throw new Exception($errorMessage);
            }

            $calledMethodReflection = new ReflectionMethod($controller, $methodName);
            /* @var $methodParamDefs ReflectionParameter[] */
            $methodParamDefs = $calledMethodReflection->getParameters();

            // TODO: udelat univerzalneji, aby se dalo pouzivat vice trid
            $methodParameters = [];
            if ($methodParamDefs[0]->getClass()->name == HttpRequest::class
                || $methodParamDefs[0]->getClass()->isSubclassOf(HttpRequest::class)
            ) {
                $methodParameters[] = $httpRequest;
            }
            // context is set
            if (isset($methodParamDefs[1])
                && ($methodParamDefs[1]->getClass()->name == Context::class
                    || $methodParamDefs[1]->getClass()->isSubclassOf(Context::class))
            ) {
                if ($routeInfo->getContext() === NULL) {
                    throw new InternalServerErrorException("Conntroller method " . $methodParamDefs[1]->getClass()->name . "::" . $calledMethodReflection->name . " expects second parameter to be Context but none gven.");
                }
                $methodParameters[] = $routeInfo->getContext();
            }

            // call controller method
            /* @var $modelAndView ModelAndView return value of controller */
            $modelAndView = $calledMethodReflection->invokeArgs($controller, $methodParameters);

            // zavola postHandle pro interceptory
            foreach ($interceptors as $interceptor) {
                $interceptor->postHandle($httpRequest, $modelAndView);
            }

            if ($modelAndView instanceof ModelAndView ) { // model and view returned by controller
                if( $modelAndView->isRedirect() ) { //is redirect
                    //handle redirect
                    $redirectControllerId = $modelAndView->getViewName();

                    if ($modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_URL) { // presmerovat na url
                        $redirectURL = $redirectControllerId;
                    } else if ($modelAndView->getRedirectType() == ModelAndView::REDIRECT_TYPE_CONTROLLER) { // presmerovat na controller
                        // najde url pro presmerovani
                        $locale = $httpRequest->getLocale();
                        if ($modelAndView->getLocale() != NULL) {
                            $locale = $modelAndView->getLocale();
                        }
                        $redirectURL = $this->urlRouter->findURL($redirectControllerId, $modelAndView->getRedirectParameters(), $modelAndView->getMethodName(), $locale);
                    } else {
                        throw new Exception("Nepodporovany typ presmerovani " . $modelAndView->getRedirectType());
                    }

                    // do redirect
                    $this->log->info("redirect to " . $redirectURL);

                    if ($modelAndView->isPermanentRedirect()) { // is permanent redirect
                        header("HTTP/1.1 301 Moved Permanently");
                    }
                    header("Location: " . $redirectURL); // send redirect
                } else { // is new
                    // vytvori view
                    $view = $this->createAndPopulateView($modelAndView, $httpRequest->getLocale());

                    // vyrendruje view
                    $view->render();
                }

                // compete view should be written by this point
                // flush data to client
                ob_flush();
                flush();
            }

            // zavola afterCompletion pro interceptory
            foreach (array_reverse($interceptors) as $interceptor) {
                $interceptor->afterCompletion($httpRequest, $modelAndView);
            }
        } catch (Exception $exception) {
            $this->handleExceptionByHandlers($exception, $httpRequest);
        } catch (Throwable $throwable) {
            $this->handleExceptionByHandlers($throwable, $httpRequest);
        }
    }

    /**
     * Handle exception by setted handllers
     * On handle - render view
     * @param Exception|Throwable $exception
     * @param null|HttpRequest $httpRequest
     * @return bool true - handle ok, false - haldler not found
     */
    protected function handleExceptionByHandlers($exception, $httpRequest) {
        if($this->config === null) { // config is not initialized at all
            require_once $this->initErrorPage;
            return true;
        }

        foreach ($this->exceptionHandlers as $exceptionHandler) {
            if($this->log) {
                $this->log->info("Trying handler ".get_class($exceptionHandler));
            }

            if ($exceptionHandler->isFor($exception)) { // found handler for exception
                if($this->log) {
                    $this->log->log(
                        $exceptionHandler->getLogLevel(),
                        "Request to url "
                            . filter_input(INPUT_SERVER, 'REQUEST_URI')
                            . " faild.",
                        ['exception' => $exception]
                    );
                }
                $modelAndView = $exceptionHandler->handleException($exception);

                if ($modelAndView instanceof ModelAndView) {//nejaky nahrobek, protože defaultni handlery nic nevrací
                    if (!$modelAndView->isRedirect()) {
                        $view = $this->createAndPopulateView($modelAndView, $httpRequest->getLocale());
                        // vyrendruje view
                        $view->render();
                    } else {
                        // TODO: sestavit presmerovani
                        $redirectControllerId = $modelAndView->getViewName();
                        // najde url pro presmerovani
                        $locale = $httpRequest->getLocale();
                        if ($modelAndView->getLocale() != NULL) {
                            $locale = $modelAndView->getLocale();
                        }
                        $redirectURL = $this->urlRouter->findURL(
                            $redirectControllerId,
                            $modelAndView->getParameters(),
                            $modelAndView->getMethodName(),
                            $locale
                        );
                        // do redirect
                        $this->log->info("redirect to " . $redirectURL);
                        header("Location: " . $redirectURL);
                    }
                }

                if (
                    $this->container !== null
                    && $this->container->has("develserver")
                    && $this->container->get("develserver")
                ) {
                    // is devel server, throw exception to be visible
                    throw $exception;
                }

                return true;
            }
        }

        if($this->log instanceof Logger) {
            $this->log->critical(
                "Application failed.",
                ['exception' => $exception]
            );
        }

        // insert error page
        require_once $this->initErrorPage;

        if (
            $this->container !== null
            && $this->container->has("develserver")
            && $this->container->get("develserver")
        ) {
            // is devel server, throw exception to be visible
            throw $exception;
        }
        return true;
    }

    protected function createAndPopulateView(ModelAndView $modelAndView, Locale $locale) {
        $this->log->info("Searching for view with name " . $modelAndView->getViewName());

        $view = NULL;
        foreach ($this->viewResolvers as $viewResolver) {
            $view = $viewResolver->buildView($modelAndView->getViewName()); // try to create view
            if ($view !== NULL) {
                break;
            }
        }

        if ($view == NULL) {
            throw new Exception("Nenalezeno potrebne view");
        }

        // provede inicializaci view
        $view->init($this->urlRouter, $modelAndView, $locale);

        return $view;
    }

    /**
     * inicializuje router pro preklad url na kontrollery
     * @param Container $container
     */
    protected function initRouter(Config $config) {
        $this->urlRouter = $config->getRouter();
    }

    /**
     * Zrapcovava http request do podoby zpracovavane dale
     *
     *
     * @return HttpRequest
     * @throws Exception
     */
    protected function populateHTTPRequest() {
        // filter_input ne vzdy funguje pro pouziti na  INPUT_SERVER
        //        $requestURL = filter_input(INPUT_SERVER, 'REQUEST_URI');
        $requestURL = $_SERVER['REQUEST_URI'];
        // najdi ? a odstran vse za nim, tj. parametry url
        $position = mb_strpos($requestURL, '?');
        if ($position !== false) {
            $requestURL = mb_substr($requestURL, 0, $position);
        }

        $locale = $this->localeResolver->resolveLocale($requestURL, $_REQUEST);
        $request = $this->localeResolver->modifyRequestURL($requestURL, $locale);

        $httpRequest = new HttpRequest($request, $_REQUEST, $locale);
        return $httpRequest;
    }

    protected function getBaseDir() {
        return $this->baseDir;
    }
}
