<?php

namespace IZON\Composer\Plugins\ModulesInstall;

use Composer\Composer;
use Composer\EventDispatcher\EventSubscriberInterface;
use Composer\IO\IOInterface;
use Composer\Plugin\PluginInterface;
use Composer\Script\Event;
use Composer\Script\ScriptEvents;
use IZON\MVC\Config;
use function IZON\Arrays\isEmpty;
use function IZON\File\deleteRecursivelly;
use function IZON\File\normalizeFileName;
use function IZON\File\normalizePath;

//use Composer\Plugin\PluginEvents;
//use Composer\Installer\PackageEvents;
//use Composer\Installer\PackageEvent;

/**
 * Provede installaci modulu do aplikace
 */
class ModulesInstall implements PluginInterface, EventSubscriberInterface
{
    /**
     * jak se jmenuje soubor obsahujici konfiguraci modulu pro praci v adminu
     */
    public const MODULE_CONFIG_FILENAME = "module-config.php";

    /**
     * @var
     */
    protected $configDir = 'config';


    /**
     * @var Composer
     */
    protected $composer;

    /**
     *
     * @var IOInterface
     */
    protected $io;

    public function activate(Composer $composer, IOInterface $io)
    {
        $this->composer = $composer;
        $this->io = $io;
    }

    public static function getSubscribedEvents()
    {
        return [
            ScriptEvents::POST_AUTOLOAD_DUMP => [
                ['onPostAutoloadDump', 0],
            ],
            "post-install-cmd" => [
                ['flushCaches', 0],
            ],
            "post-update-cmd" => [
                ['flushCaches', 0],
            ]
        ];
    }

    public function onPostAutoloadDump(Event $event)
    {
        // informace jestli je spusteno jako --dev
        //        var_dump($event->isDevMode());

        $vendorDir = $event->getComposer()->getConfig()->get('vendor-dir');
        $vendorDir = normalizePath($vendorDir);

        // nacti autoloader
        require $vendorDir . '/autoload.php';

        $baseDir = dirname($vendorDir);
        define("__BASE_DIR__", $baseDir); // basedit needed
        $config = new Config($baseDir);
        //        $config->init();

        // get ALL installed packages
        $packages = $event->getComposer()->getRepositoryManager()
            ->getLocalRepository()->getPackages();
        $installationManager = $event->getComposer()->getInstallationManager();

        // vytvorit soubor do ktereho se zapisi cesty k modulum
        // TODO: adresar pro konfigurace vrat nekde inteligentneji
        // TODO: lepsi kontrola zapisu do souboru
        $configFileName = $baseDir . "/" . $config->getConfigDir() . "/" . Config::MODULE_CONFIG_FILENAME;
        $configFile = fopen($configFileName, "w");
        flock($configFile, LOCK_EX); // aquire exclusive lock

        // zapsat havicku
        self::writeFileHeader($configFile);

        // prochazi vsechny balicky
        foreach ($packages as $package) {
            // instalalcni path modulu
            $installPath = $installationManager->getInstallPath($package);
            $installPath = normalizePath($installPath);

            //            echo $installPath ."\n";

            // konfiguracni soubor modulu
            $moduleConfigFile = $installPath . "/" . self::MODULE_CONFIG_FILENAME;
            //            echo $moduleConfigFile ."\n";

            // adresar se zdrojovimy kody
            $srcPath = $installPath . '/src';
            // adresare s dai, predpoklada se ze jsou v adresari /Dao
            $daoDirs = [];
            $daoDir = $srcPath . "/Dao";
            if (
                file_exists($daoDir)
                && is_dir($daoDir)
            ) {
                $daoDirs[] = str_replace(dirname($vendorDir), "..", $daoDir);
            }

            if (
                file_exists($moduleConfigFile) // existuje soubor konfigurace modulu, jedna se o modul admina
                || !isEmpty($daoDirs)
            ) { // nebo obsahuje nejaka daa
                $this->io->write("Writing info for module " . $package->getName());

                fwrite($configFile, "\t'" . $package->getName() . "' => [\r\n");

                if (file_exists($moduleConfigFile)) { // existuje soubor konfigurace modulu, jedna se o modul admina
                    // relativni cesta k modulu vzhledem k www/index.php jako zpousteci soubor
                    $relativeInstallPath = str_replace($vendorDir, "../vendor", $installPath);
                    fwrite($configFile, "\t\t'path' => '" . $relativeInstallPath . "'");
                }
                if (
                    file_exists($moduleConfigFile) // existuje soubor konfigurace modulu, jedna se o modul admina
                    && !isEmpty($daoDirs)
                ) {
                    fwrite($configFile, ",\r\n"); // vloz carku mezi casti
                }
                if (!isEmpty($daoDirs)) {
                    fwrite($configFile, "\t\t'daoDirs' => ['" . implode("', '", $daoDirs) . "']");
                }

                fwrite($configFile, "\r\n\t],\r\n");
            }
        }

        // projit specificke moduly a zaregistrovat je do souboru a pridat do composeru


        // uzavreni pole
        fwrite($configFile, "];");
        flock($configFile, LOCK_UN); // unlock the file
        fclose($configFile);
        $this->io->write("Modules config file created.");
    }

    protected static function writeFileHeader($configFile)
    {
        $text = "<?php\r\n"
            . "// AUTOMATICKY GENEROVANY SOUBOR PO INSTALACI MODULU\r\n"
            . "// OBSAHUJE INFORMACE O COMPOSEREM NAINSTALOVANYCH MODULECH PRO ADMINA\r\n"
            . "// cesty v 'path' jsou relativne k www/index.php\r\n"
            . "\r\n"
            . "return [\r\n";
        // TODO: kontrola jestli zapis probehl
        fwrite($configFile, $text);
    }

    /**
     * vyprazdni obsah adresare caches
     */
    public function flushCaches(Event $event)
    {
        $vendorDir = normalizeFileName($event->getComposer()->getConfig()->get('vendor-dir'));
        // try to delete old cache dir
        $cacheDir = dirname($vendorDir) . "/tmp/cache";
        if (file_exists($cacheDir)) {
            deleteRecursivelly($cacheDir);
            $this->io->write("Cache in dir " . $cacheDir . " deleted.");
        }
        // try to delete new cache dir
        $cacheDir = dirname($vendorDir) . 'app-data/tmp/cache';
        if (file_exists($cacheDir)) {
            deleteRecursivelly($cacheDir);
            $this->io->write("Cache in dir " . $cacheDir . " deleted.");
        }
    }

    public function deactivate(Composer $composer, IOInterface $io)
    {
        // TODO: Implement deactivate() method.
    }

    public function uninstall(Composer $composer, IOInterface $io)
    {
        // TODO: Implement uninstall() method.
    }
}
