<?php

namespace IZON\MessageBus\Impl;

use IZON\Logs\Logger;
use IZON\MessageBus\Message;
use IZON\MessageBus\MessageBus;
use IZON\MessageBus\MessageConsumer;
use IZON\MessageBus\MessageConsumerRegistration;

/**
 * jednoducha sbernice, ktera jen vezme zpravu a posle je vsem registrovanym consumerum,
 * aby ji zpracovali
 */
class SimpleMessageBus implements MessageBus {
    
    /**
     * @var array asociativni pole registrovanych consumeru, 
     * kde klice jsou jemena trid udalosti a hodnota je pole consumeru teto udalosti
     */
    protected $registredMessageConsumers = [];

    /**
     * @var array registrovane zpravy, ktere maji byt zpracovany
     */
    protected $registredMessages = [];
    
    /**
     * @var Logger
     */
    protected $logger;

    
    function __construct() {
        $this->logger = Logger::getLogger(__CLASS__);
    }

    /**
     * registruje consumer, ktery zpracovava zpravy se sbernice
     * @param MessageConsumer $consumer
     * @return MessageConsumerRegistration registrace na sbernici
     */
    public function simpleRegisterConsumer(MessageConsumer $consumer) {
        $consumedMessageClass = $consumer->getConsumedMessageClass();
        
        if( !isset($this->registredMessageConsumers[$consumedMessageClass]) ) {
            $this->registredMessageConsumers[$consumedMessageClasses] = [];
        }
        $this->registredMessageConsumers[$consumedMessageClass][] = $consumer;
        
        return new MessageConsumerRegistration($this, $consumer);
    }
    
    /**
     * publikuje zpravu na sbernici do defaultni fromty
     * @param Message $message
     */
    public function simplePublish(Message $message) {
        $this->registredMessages[] = $message;
    }
    
    /**
     * zpracuje zaredistrovane zpravy
     */
    public function handleRegistredMessages() {
        for($i = 0; $i < count($this->registredMessages); $i++) { // je potreba for, aby se mohly postupne pridavat i pri spusteni consumeru
            $registredMessage = $this->registredMessages[$i];
            /* @var $registredMessage Message */
            $messageClass = get_class($registredMessage);
            
            $consumers = [];
            if( isset($this->registredMessageConsumers[$messageClass]) ) {
                $consumers = $this->registredMessageConsumers[$messageClass];
            } else {
                // is not directly processed class, try to fins if is subclass of any processed class
                foreach( array_keys($this->registredMessageConsumers) as $class) {
                    if( is_subclass_of($messageClass, $class) ) {
                        $consumers = $this->registredMessageConsumers[$class];
                        // set info to hash under class
                        $this->registredMessageConsumers[$messageClass] = $consumers;
                        continue;
                    }
                }
                
                $this->logger->info("Message ". $messageClass ." is not processed by any handler");
            }
            
            foreach($consumers as $consumer) {
                $consumer->consumeMessage($registredMessage);
            }
        }
    }
}
