<?php

namespace IZON\Mailer\Impl;

use IZON\Logs\Logger;
use IZON\Mailer\Exceptions\MailException;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use PHPMailer\PHPMailer\PHPMailer;

/**
 * Iplementace maileru posilajici pres smtp
 */
class SMTPMailer implements Mailer {

    /**
     * @var string host to send emails to
     */
    protected $smtpHost;

    /**
     * @var string
     */
    protected $smtpLocalhost;

    /**
     * @var string login to server
     */
    protected $smtpLogin = NULL;

    /**
     * @var string 
     */
    protected $smtpPassword = NULL;
    
    /**
     * @var string port to communicate on, default is 25
     */
    protected $port = 25;
    
    /**
     * used keys of array: 'selector', 'domain', 'private', 'passphrase'
     * @var array
     */
    protected $DKIMSetting = [];
    
    /**
     * @var
     */
    protected $sender = NULL;

    /**
     * @deprecated is not used and probbably should not in smtp comunication class
     */
    protected $mailRenderers = [];

    /**
     * @var Logger
     */
    protected $log;

    
    public function __construct($smtpHost, $smtpLocalhost) {
        $this->smtpHost = $smtpHost;
        $this->smtpLocalhost = $smtpLocalhost;

        // sprovoznit logovani
        $this->log = Logger::getLogger(self::class);
    }

    
    public function sendMail(Mail $mail) {
        $this->log->info("Creating PHPMailer");

        $phpMailer = new PHPMailer(true);

        //Tell PHPMailer to use SMTP
        $phpMailer->isSMTP();

        //Set who the message is to be sent from
        if($mail->getFrom()['name'] != NULL) {
            $phpMailer->setFrom($mail->getFrom()['email'], $mail->getFrom()['name']);
        } else {
            $phpMailer->setFrom($mail->getFrom()['email']);
        }

        // nastaveni kam se ma email odeslat
        foreach($mail->getTo() as $to) {
            //Set who the message is to be sent to
            if($to['name'] == NULL) {
                $phpMailer->addAddress($to['email']);
            } else {
                $phpMailer->addAddress($to['email'], $to['name']);
            }
        }
        $this->log->info("Setting to");

        // nastaveni kam se ma email odeslat
        foreach($mail->getReplyTo() as $to) {
            //Set who the message is to be sent to
            if($to['name'] == NULL) {
                $phpMailer->addReplyTo($to['email']);
            } else {
                $phpMailer->addReplyTo($to['email'], $to['name']);
            }
        }
        $this->log->info("Setting reply to");

        if($this->sender != NULL) {
            $phpMailer->Sender = $this->sender;
        } else if($this->smtpLogin != NULL) {
            $phpMailer->Sender = $this->smtpLogin;
        }

        // Set the subject line
        $phpMailer->Subject = $mail->getSubject();
        // jestli se jedna o html email
        $phpMailer->isHTML($mail->isHtml());
        // nastavuje telo textu
        $phpMailer->Body = $mail->getBody();
        // nastavuje alternativni text
        if($mail->getAltBody() != '') {
            $phpMailer->AltBody = $mail->getAltBody();
        }

        // prida vsechny prilohy
        foreach($mail->getAttachments() as $attachement) {
            $phpMailer->addAttachment($attachement["path"], $attachement["name"], $attachement["encoding"], $attachement["type"]);
        }

        //set DKIM
        if(!empty($this->DKIMSetting)){
            $phpMailer->DKIM_domain = $this->DKIMSetting['domain'];
            $phpMailer->DKIM_selector = $this->DKIMSetting['selector'];
            $phpMailer->DKIM_private = $this->DKIMSetting['private'];
            $phpMailer->DKIM_passphrase = $this->DKIMSetting['passphrase'];
        }

        // TODO: nastavit codepage atd
        // Codepage UTF8
        $phpMailer->CharSet = 'utf-8';
        
        /** nastaveni parametru smtp */
        $phpMailer->Host = $this->smtpHost;
        //Set the SMTP port number - likely to be 25, 465 or 587
        $phpMailer->Port = $this->port;
        
        if($this->smtpLogin != NULL && $this->smtpPassword != NULL) {
            //Whether to use SMTP authentication
            $phpMailer->SMTPAuth = true;
            //Username to use for SMTP authentication
            $phpMailer->Username = $this->smtpLogin;
            //Password to use for SMTP authentication
            $phpMailer->Password = $this->smtpPassword;
        }
        
        // TODO: pouzit nastaveni
        $phpMailer->SMTPOptions = array('ssl' => array('verify_peer' => false, 'verify_peer_name' => false, 'allow_self_signed' => true));
        
        try {
            //send the message, check for errors
            if(!$phpMailer->send()) {
                $e = new MailException($phpMailer->ErrorInfo);
                $this->log->warning("Send email exception.", ["exception" => $e]);
                throw $e;
            }
        } catch(\PHPMailer\PHPMailer\Exception $e) {
            $e = new MailException($e);
            $this->log->warning("Send email exception.", ["exception" => $e]);
            throw $e;
        }
    }

    public function getSmtpLogin() {
        return $this->smtpLogin;
    }

    public function getSmtpPassword() {
        return $this->smtpPassword;
    }

    public function setSmtpLoginAndPassword($smtpLogin, $smtpPassword) {
        $this->smtpLogin = $smtpLogin;
        $this->smtpPassword = $smtpPassword;
    }

    /**
     * @param string nastavuje email odesilatele, ktery je viditelny jen pro smtp servery a musi existovat pro kontrolu prijimajicim mail serverem
     */
    public function setSender($sender) {
        $this->sender = $sender;
    }
    
    /**
     * sets port to comunicate on
     * @param int $port
     */
    public function setPort($port) {
        $this->port = $port;
    }

    /**
     * Set DKIm setting for this mailer
     *
     * @param array $dkimSetting Array of config params for DKIM
     *  'domain' => string - name of domain REQUIRED
     *  'selector' => string -  selector    REQUIRED
     *  'private' => string - path to file with key REQUIRED
     *  'passphrase' => string
     */
    public function setDKIMSettting(array $dkimSetting) {
        if(!empty($dkimSetting['domain']) && !empty($dkimSetting['selector']) && !empty($dkimSetting['private'])) {
            $this->DKIMSetting['domain'] = $dkimSetting['domain'];
            $this->DKIMSetting['selector'] = $dkimSetting['selector'];
            $this->DKIMSetting['private'] = $dkimSetting['private'];
            if(!empty($dkimSetting['passphrase'])){
                $this->DKIMSetting['passphrase'] = $dkimSetting['passphrase'];
            }else{
                $this->DKIMSetting['passphrase'] = '';
            }
        }
    }

    /**
     * @deprecated is not used and probbably should not in smtp comunication class
     */
    protected function renderMailBody(Mail $mail) {

    }
}

/*
Povidani od Soba k smtp


Zdar,

posilam navod, jak spravne odesilat mejly, aby nebyl problem s dorucenim, protoze se to na mnoha mistech (pokud ne vsude) porad dela blbe.

TL;DR verze: Je treba v phpmaileru nastavit Sender na nazevuctu@mailer.izon.cz.

Plna verze:

Zakladni kod pro phpmailer je nasledujici:

$mail = new PHPMailer();
$mail->IsSMTP();
$mail->Host = 'mailer.izon.cz'; // 1 - server
$mail->SMTPAuth = true;
$mail->Username = 'izon@mailer.izon.cz'; // 2 - prihlaseni
$mail->Password = '...';
$mail->From = 'news@izon.cz'; // 3 - uzivatelsky viditelny odesilatel
$mail->FromName = 'IZON News';
$mail->Sender = 'izon@mailer.izon.cz'; // 4 - smtp odesilatel
$mail->CharSet = 'utf-8';
$mail->Encoding = 'base64';
$mail->Subject = 'Test';
$mail->AddAddress('prijemce@example.net');
$mail->addReplyTo('support@izon.cz', 'Support'); // 5 - kam poslat odpoved
$mail->IsHTML(false);
$mail->Body = 'Test!';
$mail->Send();

Vysvetleni:

1) Izoni smtp server pro postu z klientskych webu je mailer.izon.cz. Pokud se jeste nekde pouziva josef.izon.cz, je treba to zmenit, idealne co nejdriv. V takovym pripade se prihlaseni meni z puvodniho mailer-neco@josef.izon.cz na neco@mailer.izon.cz, heslo zustava.

2) Prihlasovaci jmeno. Ze to ma format e-mailovy adresy, to vubec nic neznamena, je to jen uzivatelsky jmeno, nikde se s tim jako s adresou nepracuje.

3) Odesilatel, kteryho uvidi uzivatel. Jemu se budou posilat pripadny odpovedi, pokud neexistuje Reply-To.

4) Odesilatel pro smtp. Ten se pouzije pri komunikaci mezi serverama. Uzivatel ho normalne nevidi, pokud nebude hrabat v hlavickach mejlu. Tohle je ta dulezita vec, ktera ted chybi. Vysvetleni viz dal.

5) Nepovinna adresa, na kterou pujde uzivatelova odpoved. Nastavuje se pokud chceme mit ve From nejakou "peknou" adresu, ale nechceme, aby nam na ni neco chodilo.

-

Proc je dulezitej jinej odesilatel pro smtp? Protoze jsme v roce 2016 a pravidla pro postu pritvrdily. Driv se nikde nic nekontrolovalo a slo vsude cpat libovolny adresy. Chci dat jako odesilatele bill@microsoft.com? Driv zadnej problem. Ted uz ne.

Nejrozsirenejsi prekazka je SPF. Napr. budem provozovat web pro www.firma.cz a z nej budem chtit odesilat postu z info@firma.cz. Jenze postovni server bude mit pod palcem nekdo jinej a ten nastavi, ze posta z @firma.cz se muze odesilat jen z jeho mailserveru (coz je rozumny nastaveni). Pokud od nas odesleme mejl s odesilatem info@firma.cz a cilovej server kontroluje SPF, tak nas s tim posle do haje, protoze nas server mezi povolenyma pro @firma.cz nebude. Pozadovat po kazdym vyjimku je nesmysl.

Proto jako smtp odesilatele pouzijeme firma@mailer.izon.cz a problem zmizi, protoze mailer.izon.cz mame pod palcem my a odesilani z nasich serveru mame pro @mailer.izon.cz samozrejme povoleny. Uzivatel stale muze videt jako odesilatele info@firma.cz, protoze toho dame do From, kde to nicemu nevadi (to je soucast mejlu, ktera mail servery na smtp urovni nezajima).

Kdybysme to chteli jeste vylepsit, dal by se pridat DKIM podpis (at uz pomoci phpmaileru, nebo na serveru mailer.izon.cz), ale to uz se bohuzel neobejde bez spoluprace s majitelem domeny, takze se to nikomu nebude chtit delat.

--

 */