<?php

namespace IZON\MailQueue\Task;


use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Services\MailQueueService;
use IZON\Tasks\Task;
use IZON\Utils\Date;


class QueueMailSenderTask implements Task {

    /**
     * @var MailQueueService
     */
    protected $mailQueueService;

    /**
     * @var int Limit of sended mail per one run of task
     */
    protected $limitOfMailOnRun = 100;

    /**
     * @var int Num of days, half a year
     * after that num of day The mail will be deleted
     */
    protected $olderThanToDelete = 60;

    /**
     * @var bool Shall delete old emails
     */
    protected $deleteOldMails = true;

    /**
     * If true try to resend in failed state.
     * @var bool
     */
    protected $resendFailed = true;


    public function __construct(MailQueueService $mailQueueService) {
        $this->mailQueueService = $mailQueueService;
    }


    /**
     * @param int $limitOfMailOnRun
     */
    public function setLimitOfMailOnRun(int $limitOfMailOnRun) {
        $this->limitOfMailOnRun = $limitOfMailOnRun;
    }

    /**
     * Set after how much day the mail will be deleted
     * @param int|null $olderThanToDelete
     */
    public function setOlderThanToDelete(int $olderThanToDelete): void {
        $this->olderThanToDelete = $olderThanToDelete;
    }

    /**
     * Set if you want/do not wand delete old emails
     * @param bool $deleteOldMails
     */
    public function setDeleteOldMails(bool $deleteOldMails): void {
        $this->deleteOldMails = $deleteOldMails;
    }

    /**
     * Set if try to resend failed mails.
     * @param bool $resendFailed
     */
    public function setResendFailed(bool $resendFailed): void {
        $this->resendFailed = $resendFailed;
    }


    /**
     * @param array $args
     * @throws MailQueueException
     */
    public function run(array $args) {
        $failedTried = [];
        for($i=0; $i < $this->limitOfMailOnRun; $i++) {
            // find next email to send
            $queueMailId = $this->mailQueueService->getNextMailIdToSend();
            if(empty($queueMailId) && $this->resendFailed) {
                $queueMailId = $this->mailQueueService->getNextFailedMailIdToSend($failedTried);
                if(!empty($queueMailId)) {
                    $failedTried[] = $queueMailId;
                }
            }
            if( empty($queueMailId) ) {
                break;
            }
            $this->mailQueueService->sendMail($queueMailId);
        }
        $this->deleteOldMails();
    }

    /**
     * Delete old records of mails
     */
    protected function deleteOldMails() {
        if($this->deleteOldMails) {
            $date = new Date();
            $date->subTime(0, 0, 0, $this->olderThanToDelete);

            $mails = $this->mailQueueService->getOlderMailsThan($date);
            foreach($mails as $mail) {
                try {
                    $this->mailQueueService->delete($mail);
                } catch (\Exception $exception){
                    $this->getLogger()->error('Deletion of mail with id = '. $mail->getId() .' faild', ['exception' => $exception]);
                }
            }
        }
    }
}
