<?php

namespace IZON\MailQueue\Services;

use IZON\DB\DBConnection;
use IZON\DB\QueryParams\QueryParams;
use IZON\Logs\Logger;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Domain\MailQueue;
use IZON\DB\EntityManagerInterface;
use IZON\MailQueue\Domain\MailQueueAttachment;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Repositories\MailQueueRepository;
use IZON\Utils\Date;
use IZON\DB\QueryParams\OrderBy;
use function IZON\Forms\clearString;

class MailQueueService {

    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;

    /**
     * @var Mailer
     */
    protected $mailer;

    /**
     * @var MailQueueRepository
     */
    protected $mailQueueRepository;

    /**
     * Tries to resend failed emails younger than given minutes from NOW. Default is 30 mins.
     * @var int
     */
    protected $resendFailedYoungerThanMinutes = 30;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * MailQueueService constructor.
     * @param EntityManagerInterface $entityManager
     */
    public function __construct(
        EntityManagerInterface $entityManager,
        Mailer $mailer
    ) {
        $this->entityManager = $entityManager;
        $this->mailer = $mailer;

        $this->mailQueueRepository = $entityManager->getRepository(MailQueue::class);
    }

    /**
     * Tries to resend failed emails younger than given minutes from NOW. Default is 30 mins.
     * @param int $resendFailedYoungerThanMinutes
     */
    public function setResendFailedYoungerThanMinutes(int $resendFailedYoungerThanMinutes): void {
        $this->resendFailedYoungerThanMinutes = $resendFailedYoungerThanMinutes;
    }

    /**
     * @param $id
     * @return MailQueue
     */
    public function get($id) {
        $mailQueue = $this->mailQueueRepository->find($id);
        return $mailQueue;
    }

    /**
     * @return MailQueue[]
     */
    public function getAllNewInQueue(): iterable {
        $mailQueue = $this->mailQueueRepository->findBy(['status' => MailQueue::STATUS_NEW]);
        return $mailQueue;
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function save(MailQueue $mailQueue) {
        $mailQueue->setDate(new Date());
        $this->entityManager->persist($mailQueue);
        $this->entityManager->flush();
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function update(MailQueue $mailQueue) {
        $this->entityManager->persist($mailQueue);
        $this->entityManager->flush();
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function delete(MailQueue $mailQueue) {
        $this->entityManager->remove($mailQueue);
        $this->entityManager->flush();
    }

    /**
     * @param Date $date
     * @return MailQueue[]
     * @throws Exception
     */
    public function getOlderMailsThan(Date $date) {
        $mails = $this->mailQueueRepository->findOlderMailsThanInState($date->format('Y-m-d H:i:s'))->getResult();
        return $mails;
    }

    /**
     * Return id MailQueue object - mail to send
     * Return null - nothing to send
     * @return int|null
     * @throws Exception
     */
    public function getNextMailIdToSend(): ?int {
        $mailQueue = $this->mailQueueRepository->findOneBy(['status' => MailQueue::STATUS_NEW], ['id' => OrderBy::ORDER_ASC]);
        return $mailQueue === null ? null : $mailQueue->getId();
    }

    /**
     * returns id of next MailQueue object
     * @param array $excluded
     * @return int|null
     * @throws \Doctrine\ORM\NonUniqueResultException
     */
    public function getNextFailedMailIdToSend(array $excluded): ?int {
        $date = new Date();
        $date = $date->subTime(0, $this->resendFailedYoungerThanMinutes);
        $params = ['date' => $date->format('Y-m-d H:i:s')];
        if(!empty($excluded)) {
            $params['excluded'] = $excluded;
        }
        $qp = new QueryParams($params);
        $mailQueue = $this->mailQueueRepository
            ->findNextFailedMailToSend($qp)
            ->setMaxResults(1)
            ->getOneOrNullResult();


        return $mailQueue === null ? null : $mailQueue->getId();
    }

    public function sendMail(int $queueMailId) {
        /** @var MailQueue $queueMail */
        $queueMail = $this->mailQueueRepository->find($queueMailId);
        if(
            $queueMail === null
            || !in_array($queueMail->getStatus(), [MailQueue::STATUS_NEW, MailQueue::STATUS_FAIL]) // do not handle other than
        ) {
            return $queueMail->getStatus();
        }

        try {
            $queueMail->setStatus(MailQueue::STATUS_SENDING);
            $this->update($queueMail);
            $mail = $this->createMailObject($queueMail);
            $this->mailer->sendMail($mail);
            $queueMail->setStatus(MailQueue::STATUS_SENDED);
            $this->update($queueMail);
        } catch(\Throwable $exception) {
            $queueMail->setStatus(MailQueue::STATUS_FAIL);
            $queueMail->setStatusText($exception->getMessage());
            $this->update($queueMail);
            $this->getLogger()->error('Sending of mail id = ' . $queueMail->getId() . " fail", ['exception' => $exception]);
        }
    }

    /**
     * @param MailQueue $mailQueue
     * @return Mail
     * @throws MailQueueException
     */
    protected function createMailObject(MailQueue $mailQueue) {
        $mail = new Mail();
        $mail->setSubject($mailQueue->getSubject());
        $mail->setHtml($mailQueue->isHtml());
        $mail->setFrom($mailQueue->getFromMail(), $mailQueue->getFromName());
        $mail->setBody($mailQueue->getContent());
        foreach($mailQueue->getToMails() as $to){
            if(!is_array($to)) {
                continue;
            }
            $name = $to['name'];
            $email = $to['email'];
            $mail->addTo($email,$name);
        }
        if(!empty($mailQueue->getReplyToMails())) {
            foreach($mailQueue->getReplyToMails() as $to) {
                if(!is_array($to)) {
                    continue;
                }
                $name = $to['name'];
                $email = $to['email'];
                $mail->addReplyTo($email, $name);
            }
        }

        foreach($mailQueue->getAttachments() as $attachment) {
            if($attachment->getAttachmentType() == MailQueueAttachment::TYPE_EMBEDDED_IMAGE) {
                $mail->addEmbeddedImage(
                    $attachment->getAttachment(),
                    $attachment->getCid(),
                    $attachment->getName(),
                    $attachment->getMimeType(),
                    $attachment->getEncoding()
                );
            } else {
                $mail->addAttachment(
                    $attachment->getAttachment(),
                    $attachment->getName(),
                    $attachment->getMimeType(),
                    $attachment->getEncoding()
                );
            }
        }

        return $mail;
    }


    /**
     * @return Logger
     */
    protected function getLogger() {
        if(empty($this->logger)){
            $this->logger = Logger::getLogger(self::class);
        }
        return $this->logger;
    }
}
