<?php

namespace IZON\MailQueue;


use IZON\IO\File;
use IZON\IO\FileInterface;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Domain\MailQueueAttachment;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Services\MailQueueService;


class QueueMailer implements Mailer {

    protected MailQueueService $mailQueueService;

    /**
     * QueueMailer constructor.
     * @param MailQueueService $mailQueueService
     */
    public function __construct(MailQueueService $mailQueueService) {
        $this->mailQueueService = $mailQueueService;
    }


    /**
     * @param Mail $mail
     * @throws Exceptions\MailQueueException
     */
    public function sendMail(Mail $mail) {
        $queue = $this->createQueueMail($mail);
        $queue->setStatus(MailQueue::STATUS_NEW);
        $this->mailQueueService->save($queue);
    }

    /**
     * Create queue mail by mail
     * @param Mail $mail
     * @return MailQueue
     * @throws Exceptions\MailQueueException
     */
    protected function createQueueMail(Mail $mail) {
        $queue = new MailQueue();
        if(empty($mail->getFrom())){
            throw new MailQueueException('Sender of mail can not be empty');
        }
        $queue->setFromMail($mail->getFrom()['email'], (is_null($mail->getFrom()['name'])? '': $mail->getFrom()['name']));
        if(empty($mail->getSubject())) {
            $queue->setSubject("");
        }else {
            $queue->setSubject($mail->getSubject());
        }
        if(empty($mail->getBody())) {
            $queue->setContent("");
        }else {
            $queue->setContent($mail->getBody());
        }
        $queue->setHtml($mail->isHtml());
        foreach($mail->getReplyTo() as $replyTo) {
            $queue->addReplyToMails($replyTo['email'], (is_null($replyTo['name']) ? '' : $replyTo['name']));
        }
        foreach($mail->getTo() as $to){
            $queue->addToMail($to['email'], (is_null($to['name']) ? '' : $to['name']));
        }

        foreach($mail->getAttachments() as $attachment) {
            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setAttachmentType(MailQueueAttachment::TYPE_ATTACHMENT);
            $queueAttachment->setName($attachment["name"]);
            $queueAttachment->setEncoding($attachment["encoding"]);
            $queueAttachment->setMimeType($attachment["type"]);

            if( array_key_exists('file', $attachment) ) { // is file
                /** @var FileInterface $file */
                $file = $attachment["file"];
                $queueAttachment->setAttachment($file);

            } else { // is path to file
                $file = new File($attachment["path"]);
                $queueAttachment->setAttachment($file);
            }

            $queue->addAttachment($queueAttachment);
        }

        foreach($mail->getEmbeddedImages() as $embeddedImage) {
            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setAttachmentType(MailQueueAttachment::TYPE_EMBEDDED_IMAGE);
            $queueAttachment->setCid($embeddedImage['cid']);
            $queueAttachment->setName($embeddedImage['name']);
            $queueAttachment->setEncoding($embeddedImage['encoding']);
            $queueAttachment->setMimeType($embeddedImage['type']);

            if(array_key_exists('file', $embeddedImage)) { // is file
                /** @var FileInterface $file */
                $file = $embeddedImage['file'];
                $queueAttachment->setAttachment($file);
            } else { // is path to file
                $file = new File($embeddedImage['path']);
                $queueAttachment->setAttachment($file);
            }

            $queue->addAttachment($queueAttachment);
        }

        return $queue;
    }
}
