<?php

namespace IZON\MailQueue\Domain;


use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\Utils\Date;


/**
 * @author IZON s.r.o. <info@izon.cz>
 */
#[ORM\Entity]
#[ORM\Table(name: "core_mail_queue")]
class MailQueue  {

    const STATUS_NEW = 'new';

    const STATUS_SENDING = "sending";

    const STATUS_SENDED = 'sent';

    const STATUS_FAILDSEND = 'failSend';

    const STATUS_FAIL = 'fail';


    #[ORM\Id]
    #[ORM\Column]
    #[ORM\GeneratedValue]
    protected ?int $id = null;

    #[ORM\Column(type: "izon_datetime")]
    protected Date $date;

    #[ORM\Column]
    protected string $subject = "";

    #[ORM\Column(type: "text")]
    protected string $content = "";

    #[ORM\Column]
    protected bool $html = true;

    /**
     * @var array {'email' => my@ema.il, 'name' => myName}
     */
    #[ORM\Column(type: "json")]
    protected array $fromMail = [];

    /**
     * @var array [{'email' => my@ema.il, 'name' => myName}]
     */
    #[ORM\Column(type: "json")]
    protected array $toMails = [];

    /**
     * @var array [{'email' => my@ema.il, 'name' => myName}]
     */
    #[ORM\Column(type: "json")]
    protected array $replyToMails = [];

    #[ORM\Column]
    protected string $status;

    #[ORM\Column]
    protected string $statusText = "";

    /**
     * @var Collection<MailQueueAttachment>
     */
    #[ORM\OneToMany(targetEntity: MailQueueAttachment::class, mappedBy: "mailQueue", cascade: ["persist", "remove"], orphanRemoval: true)]
    protected Collection $attachments;


    public function __construct() {
        $this->attachments = new ArrayCollection();
    }


    /**
     * @return int
     */
    public function getId(): int {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getSubject(): string {
        return $this->subject;
    }

    /**
     * @return string
     */
    public function getContent(): string {
        return $this->content;
    }

    /**
     * @return bool
     */
    public function isHtml(): bool {
        return $this->html;
    }

    /**
     * @return string
     * @throws MailQueueException on fail (invalid value)
     */
    public function getFromMail(): string {
        if(!isset($this->fromMail['email'])) {
            throw new MailQueueException('Mail is not set');
        }
        return $this->fromMail['email'];
    }

    /**
     * @return string
     * @throws MailQueueException on fail (invalid value)
     */
    public function getFromName(): string {
        if(!isset($this->fromMail['name'])) {
            throw new MailQueueException('Mail is not sat');
        }
        return $this->fromMail['name'];
    }

    /**
     * @return array|null [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on fail (invalid value)
     */
    public function getToMails() {
        if(empty($this->toMails)) {
            return null;
        }
        return $this->toMails;
    }

    /**
     * @return array|null [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException
     */
    public function getReplyToMails() {
        if(empty($this->replyToMails)){
            return null;
        }
        return $this->replyToMails;
    }

    /**
     * @param string $subject
     */
    public function setSubject(string $subject): void {
        $this->subject = $subject;
    }

    /**
     * @param string $content
     */
    public function setContent(string $content): void {
        $this->content = $content;
    }

    /**
     * @param bool $html
     */
    public function setHtml(bool $html): void {
        $this->html = $html;
    }

    /**
     * @param string $name
     * @param string $email
     */
    public function setFromMail(string $email, string $name = ""): void {
        $array = ['name' => $name, 'email' => $email];
        $this->fromMail = $array;
    }

    /**
     * @param array $toMails in syntax [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on wrong syntax of input array
     */
    public function setToMails(array $toMails): void {
        foreach($toMails as $item) {
            if(!isset($item['email'])) {
                throw new MailQueueException('Mail is not sat');
            }
        }
        $this->toMails = $toMails;
    }

    /**
     * @param string $email
     * @param string $name
     * @throws MailQueueException
     */
    public function addToMail(string $email, string $name = ""){
        $array = ['name' => $name, 'email' => $email];
        $this->toMails[] = $array;
    }

    /**
     * @param array $toMails in syntax [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on wrong syntax of input array
     */
    public function setReplyToMails(array $replyToMails): void {
        foreach($replyToMails as $item) {
            if(!isset($item['email'])) {
                throw new MailQueueException('Mail is not sat');
            }
        }
        $this->replyToMails = $replyToMails;
    }

    /**
     * @param string $email
     * @param string $name
     * @throws MailQueueException
     */
    public function addReplyToMails(string $email, string $name = ""){
        $array = ['name' => $name, 'email' => $email];
        $this->replyToMails[] = $array;
    }


    /**
     * @return Collection<MailQueueAttachment>
     */
    public function getAttachments(): Collection {
        return $this->attachments;
    }

    /**
     * @param Collection<MailQueueAttachment> $attachments
     */
    public function setAttachments(Collection $attachments): void {
        foreach($attachments as $attachment) {
            $attachment->setMailQueue($this);
        }
        $this->attachments = $attachments;
    }

    /**
     * @param MailQueueAttachment $attachment
     */
    public function addAttachment(MailQueueAttachment $attachment): void {
        $attachment->setMailQueue($this);
        $this->attachments[] = $attachment;
    }

    /**
     * @return string
     */
    public function getStatus(): string {
        return $this->status;
    }

    /**
     * @param string $status
     */
    public function setStatus(string $status): void {
        $this->status = $status;
    }

    /**
     * @return string
     */
    public function getStatusText(): string {
        return $this->statusText;
    }

    /**
     * @param string $statusText
     */
    public function setStatusText(string $statusText): void {
        $this->statusText = $statusText;
    }

    /**
     * @return Date
     */
    public function getDate(): Date {
        return $this->date;
    }

    /**
     * @param Date $date
     */
    public function setDate(Date $date): void {
        $this->date = $date;
    }






}
