<?php

namespace IZON\MailQueue\Services;

use IZON\DB\DBConnection;
use IZON\DB\QueryParams\QueryParams;
use IZON\MailQueue\Domain\MailQueue;
use IZON\DB\EntityManagerInterface;
use IZON\MailQueue\Repositories\MailQueueRepository;
use IZON\Utils\Date;
use IZON\DB\QueryParams\OrderBy;
use function IZON\Forms\clearString;

class MailQueueService {

    /**
     * @var EntityManagerInterface
     */
    protected $entityManager;

    /**
     * @var MailQueueRepository
     */
    protected $mailQueueRepository;

    /**
     * Tries to resend failed emails younger than given minutes from NOW. Default is 30 mins.
     * @var int
     */
    protected $resendFailedYoungerThanMinutes = 30;

    /**
     * MailQueueService constructor.
     * @param EntityManagerInterface $entityManager
     */
    public function __construct(EntityManagerInterface $entityManager) {
        $this->entityManager = $entityManager;
        $this->mailQueueRepository = $entityManager->getRepository(MailQueue::class);
    }

    /**
     * Tries to resend failed emails younger than given minutes from NOW. Default is 30 mins.
     * @param int $resendFailedYoungerThanMinutes
     */
    public function setResendFailedYoungerThanMinutes(int $resendFailedYoungerThanMinutes): void {
        $this->resendFailedYoungerThanMinutes = $resendFailedYoungerThanMinutes;
    }

    /**
     * @param $id
     * @return MailQueue
     */
    public function get($id) {
        $mailQueue = $this->mailQueueRepository->find($id);
        return $mailQueue;
    }

    /**
     * @return MailQueue[]
     */
    public function getAllNewInQueue(): iterable {
        $mailQueue = $this->mailQueueRepository->findBy(['status' => MailQueue::STATUS_NEW]);
        return $mailQueue;
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function save(MailQueue $mailQueue) {
        $mailQueue->setDate(new Date());
        $this->entityManager->persist($mailQueue);
        $this->entityManager->flush();
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function update(MailQueue $mailQueue) {
        $this->entityManager->persist($mailQueue);
        $this->entityManager->flush();
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function delete(MailQueue $mailQueue) {
        $this->entityManager->remove($mailQueue);
        $this->entityManager->flush();
    }

    /**
     * @param Date $date
     * @return MailQueue[]
     * @throws Exception
     */
    public function getOlderMailsThan(Date $date) {
        $mails = $this->mailQueueRepository->findOlderMailsThanInState($date->format('Y-m-d H:i:s'))->getResult();
        return $mails;
    }

    /**
     * Return MailQueue object - mail to send
     * Return null - nothing to send
     * @return MailQueue|null
     * @throws Exception
     */
    public function getNextMailToSend() {
        $mailQueue = $this->mailQueueRepository->findOneBy(['status' => MailQueue::STATUS_NEW], ['id'=>OrderBy::ORDER_ASC]);
        return $mailQueue;
    }

    public function getNextFailedToSend(array $excluded) {
        $date = new Date();
        $date = $date->subTime(0, $this->resendFailedYoungerThanMinutes);
        $params = ['date' => $date->format('Y-m-d H:i:s')];
        if(!empty($excluded)) {
            $params['excluded'] = $excluded;
        }
        $qp = new QueryParams($params);
        $mailQueue = $this->mailQueueRepository
            ->findNextFailedMailToSend($qp)
            ->setMaxResults(1)
            ->getOneOrNullResult();
        return $mailQueue;
    }

}
