<?php

namespace IZON\MailQueue\Task;

use IZON\Logs\Logger;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Domain\MailQueueAttachment;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Services\MailQueueService;
use IZON\Tasks\Task;
use IZON\Utils\Date;


class QueueMailSenderTask implements Task {

    /**
     * @var Mailer
     */
    protected $mailer;

    /**
     * @var MailQueueService
     */
    protected $mailQueueService;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var int Limit of sended mail per one run of task
     */
    protected $limitOfMailOnRun = 100;

    /**
     * @var int Num of days, half a year
     * after that num of day The mail will be deleted
     */
    protected $olderThanToDelete = 180;

    /**
     * @var bool Shall delete old emails
     */
    protected $deleteOldMails = true;


    public function __construct(Mailer $mailer, MailQueueService $mailQueueService) {
        $this->mailer = $mailer;
        $this->mailQueueService = $mailQueueService;
    }


    /**
     * @param int $limitOfMailOnRun
     */
    public function setLimitOfMailOnRun(int $limitOfMailOnRun) {
        $this->limitOfMailOnRun = $limitOfMailOnRun;
    }

    /**
     * Set after how much day the mail will be deleted
     * @param int|null $olderThanToDelete
     */
    public function setOlderThanToDelete(int $olderThanToDelete): void {
        $this->olderThanToDelete = $olderThanToDelete;
    }

    /**
     * Set if you want/do not wand delete old emails
     * @param bool $deleteOldMails
     */
    public function setDeleteOldMails(bool $deleteOldMails): void {
        $this->deleteOldMails = $deleteOldMails;
    }



    /**
     * @param array $args
     * @throws MailQueueException
     */
    public function run(array $args) {
        //TODO: try to resend mails in failed state
        for($i=0; $i < $this->limitOfMailOnRun; $i++) {
            // find next email to send
            $queueMail = $this->mailQueueService->getNextMailToSend();
            if( empty($queueMail) ) {
                break;
            }

            try {
                $queueMail->setStatus(MailQueue::STATUS_SENDING);
                $this->mailQueueService->update($queueMail);
                $mail = $this->createMailObject($queueMail);
                $this->mailer->sendMail($mail);
                $queueMail->setStatus(MailQueue::STATUS_SENDED);
                $this->mailQueueService->update($queueMail);
            } catch(\Throwable $exception) {
                $queueMail->setStatus(MailQueue::STATUS_FAIL);
                $queueMail->setStatusText($exception->getMessage());
                $this->mailQueueService->update($queueMail);
                $this->getLogger()->error('Sending of mail id = ' . $queueMail->getId() . " fail", ['exception' => $exception]);
                continue;
            }
        }
        $this->deleteOldMails();
    }

    /**
     * Delete old records of mails
     */
    protected function deleteOldMails() {
        if($this->deleteOldMails) {
            $date = new Date();
            $date->subTime(0, 0, 0, $this->olderThanToDelete);

            $mails = $this->mailQueueService->getOlderMailsThan($date);
            foreach($mails as $mail) {
                try {
                    $this->mailQueueService->delete($mail);
                } catch (\Exception $exception){
                    $this->getLogger()->error('Deletion of mail with id = '. $mail->getId() .' faild', ['exception' => $exception]);
                }
            }
        }
    }

    /**
     * @param MailQueue $mailQueue
     * @return Mail
     * @throws MailQueueException
     */
    protected function createMailObject(MailQueue $mailQueue) {
        $mail = new Mail();
        $mail->setSubject($mailQueue->getSubject());
        $mail->setHtml($mailQueue->isHtml());
        $mail->setFrom($mailQueue->getFromMail(), $mailQueue->getFromName());
        $mail->setBody($mailQueue->getContent());
        foreach($mailQueue->getToMails() as $to){
            if(!is_array($to)) {
                continue;
            }
            $name = $to['name'];
            $email = $to['email'];
            $mail->addTo($email,$name);
        }
        if(!empty($mailQueue->getReplyToMails())) {
            foreach($mailQueue->getReplyToMails() as $to) {
                if(!is_array($to)) {
                    continue;
                }
                $name = $to['name'];
                $email = $to['email'];
                $mail->addReplyTo($email, $name);
            }
        }

        foreach($mailQueue->getAttachments() as $attachment) {
            $prefix = sprintf('%s:', MailQueueAttachment::TYPE_EMBEDDED_IMAGE);

            if(strpos($attachment->getName(), $prefix) === 0) {
                list($prefix, $cid, $name) = explode(':', $attachment->getName());
                $mail->addEmbeddedImage($attachment->getAttachment(), $cid, $name, $attachment->getType(), $attachment->getEncoding());
            } else {
                $mail->addAttachment($attachment->getAttachment(), $attachment->getName(), $attachment->getType(), $attachment->getEncoding());
            }
        }

        return $mail;
    }

    /**
     * @return Logger
     */
    protected function getLogger() {
        if(empty($this->logger)){
            $this->logger = Logger::getLogger(self::class);
        }
        return $this->logger;
    }


}
