<?php

namespace IZON\MailQueue;


use IZON\IO\File;
use IZON\IO\FileInterface;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Domain\MailQueueAttachment;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Services\MailQueueService;
use function IZON\Object\createGetterMethodName;


class QueueMailer implements Mailer {

    /**
     * @var MailQueueService
     */
    protected $mailQueueService;

    /**
     * QueueMailer constructor.
     * @param MailQueueService $mailQueueService
     */
    public function __construct(MailQueueService $mailQueueService) {
        $this->mailQueueService = $mailQueueService;
    }


    /**
     * @param Mail $mail
     * @throws Exceptions\MailQueueException
     */
    public function sendMail(Mail $mail) {
        $queue = $this->createQueueMail($mail);
        $queue->setStatus(MailQueue::STATUS_NEW);
        $this->mailQueueService->save($queue);
    }

    /**
     * Create queue mail by mail
     * @param Mail $mail
     * @return MailQueue
     * @throws Exceptions\MailQueueException
     */
    protected function createQueueMail(Mail $mail) {
        $queue = new MailQueue();
        if(empty($mail->getFrom())){
            throw new MailQueueException('Sender of mail can not be empty');
        }
        $queue->setFromMail($mail->getFrom()['email'], (is_null($mail->getFrom()['name'])? '': $mail->getFrom()['name']));
        if(empty($mail->getSubject())){
            $queue->setSubject("");
        }else {
            $queue->setSubject($mail->getSubject());
        }
        if(empty($mail->getBody())){
            $queue->setContent("");
        }else {
            $queue->setContent($mail->getBody());
        }
        $queue->setHtml($mail->isHtml());
        foreach($mail->getReplyTo() as $to){
            $queue->addReplyToMails($to['email'], (is_null($to['name']) ? '' : $to['name']));
        }
        foreach($mail->getTo() as $to){
            $queue->addToMail($to['email'], (is_null($to['name']) ? '' : $to['name']));
        }

        foreach($mail->getAttachments() as $attachment) {
            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setName($attachment["name"]);
            $queueAttachment->setEncoding($attachment["encoding"]);
            $queueAttachment->setType($attachment["type"]);

            if( array_key_exists('file', $attachment) ) { // is file
                /** @var FileInterface $file */
                $file = $attachment["file"];
                $queueAttachment->setAttachment($file);

            } else { // is path to file
                $file = new File($attachment["path"]);
                $queueAttachment->setAttachment($file);
            }

            $queue->addAttachment($queueAttachment);
        }

        foreach($mail->getEmbeddedImages() as $embeddedImage) {
            $name = sprintf('%s:%s:%s', MailQueueAttachment::TYPE_EMBEDDED_IMAGE, $embeddedImage['cid'], $embeddedImage['name']);

            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setName($name);
            $queueAttachment->setEncoding($embeddedImage['encoding']);
            $queueAttachment->setType($embeddedImage['type']);

            if(array_key_exists('file', $embeddedImage)) { // is file
                /** @var FileInterface $file */
                $file = $embeddedImage['file'];
                $queueAttachment->setAttachment($file);
            } else { // is path to file
                $file = new File($embeddedImage['path']);
                $queueAttachment->setAttachment($file);
            }

            $queue->addAttachment($queueAttachment);
        }

        return $queue;
    }
}
