<?php

namespace IZON\MailQueue\Domain;

use Doctrine\ORM\Mapping as ORM;
use IZON\DB\DBObject;
use function IZON\DB\transientField;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\Utils\Date;
use IZON\IO\Image;
use IZON\IO\File;

/**
 * Description of MailQueue
 *
 * @ORM\Entity()
 * @ORM\Table(name="core_mail_queue")
 * @author IZON s.r.o. <info@izon.cz>
 */
class MailQueue  {

    const STATUS_NEW = 'new';

    const STATUS_SENDING = "sending";

    const STATUS_SENDED = 'sent';

    const STATUS_FAILDSEND = 'failSend';

    const STATUS_FAIL = 'fail';

    /**
     * @ORM\Id()
     * @ORM\Column()
     * @ORM\GeneratedValue()
     * @var integer
     */
    protected $id;

    /**
     * @ORM\Column(type="izon_datetime")
     * @var Date
     */
    protected $date;

    /**
     * @ORM\Column(type='text')
     * @var string
     */
    protected $subject = "";

    /**
     * @ORM\Column(type='text')
     * @var string
     */
    protected $content = "";

    /**
     * @ORM\Column()
     * @var boolean
     */
    protected $html = True;

    /**
     * @ORM\Column(type="json")
     * @var string json {'email' => my@ema.il, 'name' => myName}
     */
    protected $fromMail = "";

    /**
     * @ORM\Column(type="json")
     * @var string json  [{'email' => my@ema.il, 'name' => myName}]
     */
    protected $toMails = "";

    /**
     * @ORM\Column(type="json")
     * @var string json json [{'email' => my@ema.il, 'name' => myName}]
     */
    protected $replyToMails = "";

    /**
     * @ORM\Column()
     * @var string
     */
    protected $status;

    /**
     * @ORM\Column()
     * @var string
     */
    protected $statusText = "";

    /**
     * @ORM\OneToMany(targetEntity="MailQueueAttachment", mappedBy="mailQueue", cascade={"persist", "remove"}, orphanRemoval=true)
     * @var MailQueueAttachment[]|iterable
     */
    protected $attachments = [];


    public function __toString() {
        return "Id:`".$this->id ."`".";"."Subject:`".$this->subject."`".";"."Content:`".$this->content."`".";"."Html:`".$this->html."`".";"."FromMail:`".$this->fromMail."`".";"."ToMails:`".$this->toMails."`".";"."replyToMails:`".$this->replyToMails."`";
    }

    /**
     * @return int
     */
    public function getId(): int {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getSubject(): string {
        return $this->subject;
    }

    /**
     * @return string
     */
    public function getContent(): string {
        return $this->content;
    }

    /**
     * @return bool
     */
    public function isHtml(): bool {
        return $this->html;
    }

    /**
     * @return string
     * @throws MailQueueException on fail (invalid value)
     */
    public function getFromMail(): string {
        if(empty($this->fromMail)){
            return "";
        }
        $array = json_decode($this->fromMail, true);
        if(is_null($array)) {
            throw new MailQueueException('Can not read from mail');
        }
        if(!isset($array['email'])) {
            throw new MailQueueException('Mail is not sat');
        }
        return $array['email'];
    }

    /**
     * @return string
     * @throws MailQueueException on fail (invalid value)
     */
    public function getFromName(): string {
        if(empty($this->fromMail)){
            return "";
        }
        $array = json_decode($this->fromMail, true);
        if(is_null($array)) {
            throw new MailQueueException('Can not read from mail');
        }
        if(!isset($array['name'])) {
            throw new MailQueueException('Mail is not sat');
        }
        return $array['name'];
    }

    /**
     * @return array|null [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on fail (invalid value)
     */
    public function getToMails() {
        if(empty($this->toMails)){
            return null;
        }
        $array = json_decode($this->toMails, true);
        if(is_null($array)) {
            throw new MailQueueException('Can not read from mail');
        }
        return $array;
    }

    /**
     * @return array|null [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException
     */
    public function getReplyToMails(){
        if(empty($this->replyToMails)){
            return null;
        }
        $array = json_decode($this->replyToMails, true);
        if(is_null($array)) {
            throw new MailQueueException('Can not read from mail');
        }
        return $array;
    }

    /**
     * @param string $subject
     */
    public function setSubject(string $subject): void {
        $this->subject = $subject;
    }

    /**
     * @param string $content
     */
    public function setContent(string $content): void {
        $this->content = $content;
    }

    /**
     * @param bool $html
     */
    public function setHtml(bool $html): void {
        $this->html = $html;
    }

    /**
     * @param string $name
     * @param string $email
     */
    public function setFromMail(string $email, string $name = ""): void {
        $array = ['name' => $name, 'email' => $email];
        $this->fromMail = json_encode($array);
    }

    /**
     * @param array $toMails in syntax [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on wrong syntax of input array
     */
    public function setToMails(array $toMails): void {
        foreach($toMails as $item) {
            if(!isset($item['email'])) {
                throw new MailQueueException('Mail is not sat');
            }
        }
        $this->toMails = json_encode($toMails);
    }

    /**
     * @param string $email
     * @param string $name
     * @throws MailQueueException
     */
    public function addToMail(string $email, string $name = ""){
        $array = ['name' => $name, 'email' => $email];
        $toMails = $this->getToMails();
        $toMails[] = $array;
        $this->setToMails($toMails);
    }

    /**
     * @param array $toMails in syntax [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on wrong syntax of input array
     */
    public function setReplyToMails(array $replyToMails): void {
        foreach($replyToMails as $item) {
            if(!isset($item['email'])) {
                throw new MailQueueException('Mail is not sat');
            }
        }
        $this->replyToMails = json_encode($replyToMails);
    }

    /**
     * @param string $email
     * @param string $name
     * @throws MailQueueException
     */
    public function addReplyToMails(string $email, string $name = ""){
        $array = ['name' => $name, 'email' => $email];
        $toMails = $this->getReplyToMails();
        $toMails[] = $array;
        $this->setReplyToMails($toMails);
    }


    /**
     * @return MailQueueAttachment[]
     */
    public function getAttachments(): iterable {
        return $this->attachments;
    }

    /**
     * @param MailQueueAttachment[] $attachments
     */
    public function setAttachments(iterable $attachments): void {
        foreach($attachments as $attachment) {
            $attachment->setMailQueue($this);
        }
        $this->attachments = $attachments;
    }

    /**
     * @param MailQueueAttachment $attachment
     */
    public function addAttachment(MailQueueAttachment $attachment): void {
        $attachment->setMailQueue($this);
        $this->attachments[] = $attachment;
    }

    /**
     * @return string
     */
    public function getStatus(): string {
        return $this->status;
    }

    /**
     * @param string $status
     */
    public function setStatus(string $status): void {
        $this->status = $status;
    }

    /**
     * @return string
     */
    public function getStatusText(): string {
        return $this->statusText;
    }

    /**
     * @param string $statusText
     */
    public function setStatusText(string $statusText): void {
        $this->statusText = $statusText;
    }

    /**
     * @return Date
     */
    public function getDate(): Date {
        return $this->date;
    }

    /**
     * @param Date $date
     */
    public function setDate(Date $date): void {
        $this->date = $date;
    }






}
