<?php

namespace IZON\MailQueue\Task;


use IZON\Logs\Logger;
use IZON\MailQueue\Task\Services\TaskMailQueueService;
use IZON\MailQueue\Web\Services\MailQueueService;
use IZON\Tasks\Task;
use IZON\Utils\Date;


class MailQueueSenderTask implements Task {

    protected TaskMailQueueService $mailQueueService;

    /**
     * @var int Limit of mail to send per one run of task
     */
    protected int $limitOfMailOnRun = 60;

    /**
     * @var int Num of days, half a year
     * after that num of day The mail will be deleted
     */
    protected int $olderThanToDelete = 60;

    /**
     * @var bool Shall delete old emails
     */
    protected bool $deleteOldMails = true;

    /**
     * If true try to resend in failed state.
     * @var bool
     */
    protected bool $resendFailed = true;

    /**
     * Tries to resend failed emails younger than given minutes from NOW. Default is 30 mins.
     * @var int
     */
    protected int $resendFailedYoungerThanMinutes = 30;


    public function __construct(TaskMailQueueService $mailQueueService) {
        $this->mailQueueService = $mailQueueService;
    }

    /**
     * @param int $limitOfMailOnRun
     */
    public function setLimitOfMailOnRun(int $limitOfMailOnRun) {
        $this->limitOfMailOnRun = $limitOfMailOnRun;
    }

    /**
     * Set after how much day the mail will be deleted
     * @param int $olderThanToDelete
     */
    public function setOlderThanToDelete(int $olderThanToDelete): void {
        $this->olderThanToDelete = $olderThanToDelete;
    }

    /**
     * Set if you want/do not wand delete old emails
     * @param bool $deleteOldMails
     */
    public function setDeleteOldMails(bool $deleteOldMails): void {
        $this->deleteOldMails = $deleteOldMails;
    }

        /**
     * @param bool $resendFailed
     */
    public function setResendFailed(bool $resendFailed): void {
        $this->resendFailed = $resendFailed;
    }

    /**
     * @param int $resendFailedYoungerThanMinutes
     */
    public function setResendFailedYoungerThanMinutes(int $resendFailedYoungerThanMinutes): void {
        $this->resendFailedYoungerThanMinutes = $resendFailedYoungerThanMinutes;
    }

    /**
     * @param array $args
     */
    public function run(array $args) {
        /** @var int $resendFailedMailIdes ides od failed mails that where tried to resend */
        $resendFailedMailIdes = [];
        for($i=0; $i < $this->limitOfMailOnRun; $i++) {
            // find next email to send
            $queueMailId = $this->mailQueueService->getNextMailIdToSend();
            if( $queueMailId === null && $this->resendFailed ) {
                $queueMailId = $this->mailQueueService->getNextFailedMailIdToSend(
                    $resendFailedMailIdes,
                    $this->resendFailedYoungerThanMinutes
                );
                if($queueMailId !== null) {
                    $resendFailedMailIdes[] = $queueMailId;
                }
            }

            if( $queueMailId === null) { // no more emails to process
                break;
            }

            try {
                $queueMailStatus = $this->mailQueueService->sendMail($queueMailId);
            } catch(\Throwable $exception) {
                $this->mailQueueService->logSenderError(
                    'Sending of mail id = '. $queueMailId ." fail with unrecoverable error",
                    ['exception' => $exception]
                );
            }
        }
        $this->deleteOldMails();
    }

    /**
     * Delete old records of mails
     */
    protected function deleteOldMails() {
        if( $this->deleteOldMails ) {
            $date = new Date();
            $date->subTime(0, 0, 0, $this->olderThanToDelete);

            $mailIdes = $this->mailQueueService->getMailIdesOlderThan($date);
            foreach($mailIdes as $mailId) {
                try {
                    $this->mailQueueService->deleteMail($mailId);
                } catch (\Exception $exception){
                    $this->mailQueueService->logSenderError(
                        'Deletion of mail with id = '. $mailId .' faild',
                        ['exception' => $exception]
                    );
                }
            }
        }
    }
}
