<?php

namespace IZON\MailQueue\Domain;

use IZON\DB\DBObject;
use function IZON\DB\jsonField;
use function IZON\DB\transientField;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\Utils\Date;
use IZON\IO\Image;
use IZON\IO\File;

/**
 * Description of MailQueue
 * @author IZON s.r.o. <info@izon.cz>
 */
class MailQueue implements DBObject {

    public static function __tableName(): string { return 'core_mail_queues'; }

    /**
     * newly added email
     */
    const STATUS_NEW = 'new';

    /**
     * email is sending at this time
     */
    const STATUS_SENDING = "sending";

    /**
     * email was sent correctly
     */
    const STATUS_SENT = 'sent';

    /**
     *
     */
    const STATUS_SEND_FAILED = 'send-failed';

    const STATUS_FAILED = 'failed';

    /**
     * statuses to be considered for sending email
     */
    const TO_SEND_STATUSES = [
        self::STATUS_NEW,
        self::STATUS_FAILED,
        self::STATUS_SEND_FAILED,
    ];

    /**
     * statuses to be considered as email sending failed
     */
    const FAILED_STATUSES = [
        self::STATUS_FAILED,
        self::STATUS_SEND_FAILED,
    ];

    protected ?int $id = null;

    protected Date $date;

    protected string $subject= "";

    protected string $body = "";

    /**
     * @var bool if true body is html
     */
    protected bool $html = true;

    /**
     * @var array json {'email' => my@ema.il, 'name' => myName}
     */
    protected array $fromMail = [];
    public static function __fromMail() { return jsonField(); }

    /**
     * @var array json  [{'email' => my@ema.il, 'name' => myName}]
     */
    protected array $toMails = [];
    public static function __toMails() { return jsonField(); }

    /**
     * @var array json json [{'email' => my@ema.il, 'name' => myName}]
     */
    protected array $replyToMails = [];
    public static function __replyToMails() { return jsonField(); }

    /**
     * @var string
     */
    protected string $status = self::STATUS_NEW;

    /**
     * @var string
     */
    protected string $statusText = "";

    /**
     * @var MailQueueAttachment[]
     */
    protected array $attachments = [];
    public static function __attachments() { return transientField(); }

    /**
     * @return int
     */
    public function getId(): ?int {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getSubject(): string {
        return $this->subject;
    }

    /**
     * @return string
     */
    public function getBody(): string {
        return $this->body;
    }

    /**
     * @return bool
     */
    public function isHtml(): bool {
        return $this->html;
    }

    /**
     * @return null|string
     * @throws MailQueueException on fail (invalid value)
     */
    public function getFromMail(): ?string {
        if(empty($this->fromMail)) {
            return "";
        }
        if(!isset($this->fromMail['email'])) {
            throw new MailQueueException('Mail is not set');
        }
        return $this->fromMail['email'];
    }

    /**
     * @return string
     * @throws MailQueueException on fail (invalid value)
     */
    public function getFromName(): string {
        if(empty($this->fromMail)) {
            return "";
        }
        if(!isset($this->fromMail['name'])) {
            throw new MailQueueException('Mail is not set');
        }
        return $this->fromMail['name'];
    }

    /**
     * @return array [{'email' => my@ema.il, 'name' => myName}]
     */
    public function getToMails(): array {
        return $this->toMails;
    }

    /**
     * @return array [{'email' => my@ema.il, 'name' => myName}]
     */
    public function getReplyToMails(): array {
        return $this->replyToMails;
    }

    /**
     * @param string $subject
     */
    public function setSubject(string $subject): void {
        $this->subject = $subject;
    }

    /**
     * @param string $body
     */
    public function setBody(string $body): void {
        $this->body = $body;
    }

    /**
     * @param bool $html
     */
    public function setHtml(bool $html): void {
        $this->html = $html;
    }

    /**
     * @param string $name
     * @param string $email
     */
    public function setFromMail(string $email, string $name = ""): void {
        $this->fromMail = ['name' => $name, 'email' => $email];
    }

    /**
     * @param array $toMails in syntax [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on wrong syntax of input array
     */
    public function setToMails(array $toMails): void {
        foreach($toMails as $item) {
            if(!isset($item['email'])) {
                throw new MailQueueException('Mail is not sat');
            }
        }
        $this->toMails = $toMails;
    }

    /**
     * @param string $email
     * @param string $name
     * @throws MailQueueException
     */
    public function addToMail(string $email, string $name = "") {
        $array = ['name' => $name, 'email' => $email];
        $toMails = $this->getToMails();
        $toMails[] = $array;
        $this->setToMails($toMails);
    }

    /**
     * @param array $replyToMails in syntax [{'email' => my@ema.il, 'name' => myName}]
     * @throws MailQueueException on wrong syntax of input array
     */
    public function setReplyToMails(array $replyToMails): void {
        foreach($replyToMails as $item) {
            if(!isset($item['email'])) {
                throw new MailQueueException('Mail is not sat');
            }
        }
        $this->replyToMails = $replyToMails;
    }

    /**
     * @param string $email
     * @param string $name
     * @throws MailQueueException
     */
    public function addReplyToMails(string $email, string $name = ""){
        $this->replyToMails = array_merge(
            $this->replyToMails,
            [['name' => $name, 'email' => $email]]
        );
    }

    /**
     * @return MailQueueAttachment[]
     */
    public function getAttachments(): array {
        return $this->attachments;
    }

    /**
     * @param MailQueueAttachment[] $attachments
     */
    public function setAttachments(array $attachments): void {
        $this->attachments = $attachments;
    }

    /**
     * @param MailQueueAttachment $attachment
     */
    public function addAttachment(MailQueueAttachment $attachment): void {
        $this->attachments[] = $attachment;
    }

    /**
     * @return string
     */
    public function getStatus(): string {
        return $this->status;
    }

    /**
     * @param string $status
     */
    public function setStatus(string $status): void {
        $this->status = $status;
    }

    /**
     * @return string
     */
    public function getStatusText(): string {
        return $this->statusText;
    }

    /**
     * @param string $statusText
     */
    public function setStatusText(string $statusText): void {
        $this->statusText = $statusText;
    }

    /**
     * @return Date
     */
    public function getDate(): Date {
        return $this->date;
    }

    /**
     * @param Date $date
     */
    public function setDate(Date $date): void {
        $this->date = $date;
    }

}
