<?php

namespace IZON\MailQueue\Task\Services;

use Exception;
use IZON\DB\OrderBy;
use IZON\DB\QueryResult;
use IZON\Logs\Logger;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Dao\MailQueueAttachmentDao;
use IZON\MailQueue\Dao\MailQueueDao;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Domain\MailQueueAttachment;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Task\DTO\MailQueueStatus;
use IZON\MailQueue\Task\DTO\QueueMailStatus;
use IZON\Utils\Date;
use Throwable;

class TaskMailQueueService {

    protected Mailer $mailer;

    protected MailQueueDao $mailQueueDao;

    protected MailQueueAttachmentDao $mailQueueAttachmentDao;

    protected Logger $logger;

    /**
     * BaseMailQueueService constructor.
     * @param MailQueueDao $mailQueueDao
     * @param MailQueueAttachmentDao $mailQueueAttachmentDao
     */
    public function __construct(
        Mailer $mailer,
        MailQueueDao $mailQueueDao,
        MailQueueAttachmentDao $mailQueueAttachmentDao,
        string $loggerChannelName = 'mail-queue-sender'
    ) {
        $this->mailer = $mailer;
        $this->mailQueueDao = $mailQueueDao;
        $this->mailQueueAttachmentDao = $mailQueueAttachmentDao;

        $this->logger = Logger::getLogger($loggerChannelName);
    }

    /**
     * Return id MailQueue object - mail to send
     * Return null - nothing to send
     * @return int|null
     */
    public function getNextMailIdToSend(): ?int {
        /** @var null|MailQueue $mailQueue */
        $mailQueue = $this->mailQueueDao->find(['status' => MailQueue::STATUS_NEW], ['id' => OrderBy::ASC])
            ->setMaxResults(1)->uniqueResult();
        return $mailQueue === null ? null : $mailQueue->getId();
    }

    /**
     * returns id of next MailQueue object
     * @param array $excluded
     * @return int|null
     */
    public function getNextFailedMailIdToSend(array $excluded, int $resendFailedYoungerThanMinutes): ?int {
        $date = new Date();
        $date->subTime(0, $resendFailedYoungerThanMinutes);
        $params = ['date' => $date->format('Y-m-d H:i:s')];
        if(!empty($excluded)) {
            $params['excluded'] = $excluded;
        }

        /** @var null|MailQueue $mailQueue */
        $mailQueue = $this->mailQueueDao
            ->customFindNextFailedMailToSend($params)
            ->setMaxResults(1)
            ->uniqueResult();

        return $mailQueue === null ? null : $mailQueue->getId();
    }

    /**
     * @param int $queueMailId
     * @return MailQueueStatus
     * @throws Exception
     */
    public function sendMail(int $queueMailId): MailQueueStatus {
        /** @var null|MailQueue $queueMail */
        $queueMail = $this->mailQueueDao->findLockedById($queueMailId)->uniqueResult();
        if(
            $queueMail === null
            || !in_array($queueMail->getStatus(),MailQueue::TO_SEND_STATUSES) // do not handle other than statuses to send
        ) {
            return new MailQueueStatus(
                $queueMailId,
                MailQueueStatus::STATUS_SKIPPED,
                'Does not exist or not in one of MailQueue::TO_SEND_STATUSES'
            );
        }

        /** @var MailQueueAttachment[] $attachments */
        $attachments = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $queueMail->getId()])->listResult();
        $queueMail->setAttachments($attachments);

        $this->logger->info('Sending mail id = '. $queueMailId .'.');

        try {
            $queueMail->setStatus(MailQueue::STATUS_SENDING);
            $this->mailQueueDao->update($queueMail);
            $mail = $this->createMail($queueMail);
            $this->mailer->sendMail($mail);

            $queueMail->setStatus(MailQueue::STATUS_SENT);
            $this->mailQueueDao->update($queueMail);

            $this->logger->info('Mail id = '. $queueMailId .' sent.');
        } catch(Throwable $exception) {
            $queueMail->setStatus(MailQueue::STATUS_FAILED);
            $queueMail->setStatusText($exception->getMessage());
            $this->mailQueueDao->update($queueMail);

            $this->logger->error('Sending of mail id = '. $queueMailId ." fail.", ['exception' => $exception]);
        }

        return new MailQueueStatus(
            $queueMail->getId(),
            $queueMail->getStatus(),
            $queueMail->getStatusText()
        );
    }


    /**
     * @param MailQueue $mailQueue
     * @return Mail
     */
    protected function createMail(MailQueue $mailQueue): Mail {
        $mail = new Mail();
        $mail->setSubject($mailQueue->getSubject());
        $mail->setHtml($mailQueue->isHtml());
        $mail->setFrom($mailQueue->getFromMail(), $mailQueue->getFromName());
        $mail->setBody($mailQueue->getBody());
        foreach($mailQueue->getToMails() as $to){
            if(!is_array($to)) {
                continue;
            }
            $name = $to['name'];
            $email = $to['email'];
            $mail->addTo($email,$name);
        }
        if(!empty($mailQueue->getReplyToMails())) {
            foreach($mailQueue->getReplyToMails() as $to) {
                if(!is_array($to)) {
                    continue;
                }
                $name = $to['name'];
                $email = $to['email'];
                $mail->addReplyTo($email, $name);
            }
        }
        foreach($mailQueue->getAttachments() as $attachment) {
            if( $attachment->getCid() !== null ) {
                $mail->addEmbeddedImage(
                    $attachment->getAttachment()->getFsPath(),
                    $attachment->getCid(),
                    $attachment->getName(),
                    $attachment->getAttachment()->getMimeType()
                );
            } else {
                $mail->addAttachment(
                    $attachment->getAttachment()->getFsPath(),
                    $attachment->getName(),
                    $attachment->getAttachment()->getMimeType()
                );
            }
        }
        return $mail;
    }


    /**
     * get mail queue ides older than $date
     * @param Date $date
     * @return int[]
     */
    public function getMailIdesOlderThan(Date $date): array {
        $mails = $this->mailQueueDao->findMailsOlderThan($date->format('Y-m-d H:i:s'))->listResult(
            QueryResult::FETCH_ASSOC
        );
        return array_map(
            function($mailArray) {
                return $mailArray['id'];
            },
            $mails
        );
    }

    /**
     * @param int $mailId
     */
    public function deleteMail(int $mailId): void {
        /** @var null|MailQueue $mail */
        $mail = $this->mailQueueDao->find(['id' => $mailId])->uniqueResult();
        if( $mail === null ) {
            return;
        }

        $attachments = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $mailId])->listResult();
        foreach($attachments as $attachment) {
            $this->mailQueueAttachmentDao->delete($attachment);
        }
        $this->mailQueueDao->delete($mail);
    }

    public function logSenderError(string $message, array $context): void {
        $this->logger->error($message, $context);
    }
}
