<?php

namespace IZON\MailQueue\Services;

use IZON\DB\DBConnection;
use IZON\IO\File;
use IZON\Mailer\Mail;
use IZON\MailQueue\Dao\MailQueueAttachmentDao;
use IZON\MailQueue\Dao\MailQueueDao;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Domain\MailQueueAttachment;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\Utils\Date;


class MailQueueService {

    protected DBConnection $dbConnection;

    /**
     * @var string
     */
    protected string $tmpDir;

    /**
     * @var string[] temporary files to delete after MailQueue saved
     */
    protected array $tempFilePathsToDelete = [];

    protected MailQueueDao $mailQueueDao;

    protected MailQueueAttachmentDao $mailQueueAttachmentDao;


    /**
     * @param string $tmpDir
     * @param MailQueueDao $mailQueueDao
     * @param MailQueueAttachmentDao $mailQueueAttachmentDao
     */
    public function __construct(
        DBConnection $dbConnection,
        string $tmpDir,
        MailQueueDao $mailQueueDao,
        MailQueueAttachmentDao $mailQueueAttachmentDao
    ) {
        $this->dbConnection = $dbConnection;
        $this->tmpDir = $tmpDir;
        $this->mailQueueDao = $mailQueueDao;
        $this->mailQueueAttachmentDao = $mailQueueAttachmentDao;
    }


    /**
     * @param MailQueue $mailQueue
     */
    protected function fill(MailQueue $mailQueue): void {
        /** @var MailQueueAttachment[] $attachments */
        $attachments = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $mailQueue->getId()])->listResult();
        $mailQueue->setAttachments($attachments);
    }

    /**
     * @param int $id
     * @return MailQueue
     */
    public function get(int $id): MailQueue {
        return $this->executeTransactional(function() use($id) {
            /** @var MailQueue $mailQueue */
            $mailQueue = $this->mailQueueDao->load($id);
            $this->fill($mailQueue);
            return $mailQueue;
        });
    }

    /**
     * @param Mail $mail
     */
    public function send(Mail $mail): void {
        $mailQueue = $this->createQueueMail($mail);

        $this->executeTransactional(function() use ($mailQueue) {
            $mailQueue->setDate(new Date());
            $this->mailQueueDao->save($mailQueue);
            foreach($mailQueue->getAttachments() as $item){
                $item->setFkMailQueueId($mailQueue->getId());
                $this->mailQueueAttachmentDao->save($item);
            }
        });

        foreach($this->tempFilePathsToDelete as $filePathToDelete) {
            unlink($filePathToDelete);
        }
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function update(MailQueue $mailQueue): void {
        $this->executeTransactional(function() use ($mailQueue) {
            $this->mailQueueDao->update($mailQueue);
            $this->updateAttachments($mailQueue);
        });
    }


    /**
     * To update attachment - without transaction
     * @param MailQueue $mailQueue
     */
    protected function updateAttachments(MailQueue $mailQueue): void {
        /** @var MailQueueAttachment[] $olds */
        $olds = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $mailQueue->getId()])->listResult();
        $oldsHash = [];
        foreach($olds as $old) {
            $oldsHash[$old->getId()] = $old;
        }
        foreach($mailQueue->getAttachments() as $object) {
            if(isset($oldsHash[$object->getId()])){
                unset($oldsHash[$object->getId()]);
            } else {
                $object->setFkMailQueueId($mailQueue->getId());
                $this->mailQueueAttachmentDao->save($object);
            }
        }
        foreach($oldsHash as $toDelete){
            $this->mailQueueAttachmentDao->delete($toDelete);
        }
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function delete(MailQueue $mailQueue): void {
        $this->executeTransactional(function() use ($mailQueue) {
            $attachments = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $mailQueue->getId()])->listResult();
            foreach($attachments as $attachment) {
                $this->mailQueueAttachmentDao->delete($attachment);
            }
            $this->mailQueueDao->delete($mailQueue);
        });
    }

    public function setDBConnection(DBConnection $dbConnection) {
        $this->dbConnection = $dbConnection;
    }

    /**
     * executes $transactional callable in transaction (creates new transaction if transaction not opened yet)
     * @param callable $transactional
     * @return mixed value returned from transactional callable
     */
    protected function executeTransactional(callable $transactional) {
        $retVal = null;

        $transactionStarted = false;
        if( !$this->dbConnection->inTransaction() ) { // transaction not opened yet
            $this->dbConnection->beginTransaction(); // start transaction
            $transactionStarted = true;
        }

        // execute transactional
        $retVal = $transactional();

        if( $transactionStarted ) {
            $this->dbConnection->commit(); // commit started transaction
        }

        return $retVal;
    }

    /**
     * Create queue mail by mail
     * @param Mail $mail
     * @return MailQueue
     */
    protected function createQueueMail(Mail $mail): MailQueue {
        $queue = new MailQueue();
        if(empty($mail->getFrom())) {
            throw new MailQueueException('Sender of mail can not be empty');
        }
        $queue->setFromMail($mail->getFrom()['email'], (is_null($mail->getFrom()['name'])? '': $mail->getFrom()['name']));
        if(empty($mail->getSubject())){
            $queue->setSubject("");
        }else {
            $queue->setSubject($mail->getSubject());
        }
        if(empty($mail->getBody())){
            $queue->setBody("");
        }else {
            $queue->setBody($mail->getBody());
        }
        $queue->setHtml($mail->isHtml());
        foreach($mail->getReplyTo() as $to){
            $queue->addReplyToMails($to['email'], (is_null($to['name']) ? '' : $to['name']));
        }
        foreach($mail->getTo() as $to){
            $queue->addToMail($to['email'], (is_null($to['name']) ? '' : $to['name']));
        }

        foreach($mail->getAttachments() as $attachment) {
            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setName($attachment["name"]);

            $filePath = $attachment["path"];
            if( $attachment['type'] == 'content' ) {
                $filePath = $this->writeAttachmentContentToFile($attachment['name'], $attachment['content']);
                $this->tempFilePathsToDelete[] = $filePath;
            }
            $queueAttachment->setAttachment(new File($filePath));

            $queue->addAttachment($queueAttachment);
        }

        foreach($mail->getEmbeddedImages() as $embeddedImage) {
            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setName($embeddedImage["name"]);
            $queueAttachment->setCid($embeddedImage["cid"]);

            $filePath = $embeddedImage["path"];
            if( $embeddedImage['type'] == 'content' ) {
                $filePath = $this->writeAttachmentContentToFile($embeddedImage['name'], $embeddedImage['content']);
                $this->tempFilePathsToDelete[] = $filePath;
            }
            $queueAttachment->setAttachment(new File($filePath));

            $queue->addAttachment($queueAttachment);
        }

        return $queue;
    }

    /**
     * @param string $fileName
     * @param $fileContent
     * @return string path where is file stored
     */
    protected function writeAttachmentContentToFile(string $fileName, $fileContent): string {
        if( !file_exists($this->tmpDir) ) {
            mkdir($this->tmpDir, 0777, true);
        }

        $filePath = $this->tmpDir ."/". round(microtime(true) * 1000) .'-'. uniqid() .'-'. $fileName;
        file_put_contents($filePath, $fileContent);
        return $filePath;
    }
}
