<?php

namespace IZON\MailQueue\Task\Services;

use Exception;
use IZON\DB\OrderBy;
use IZON\DB\QueryResult;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Dao\MailQueueAttachmentDao;
use IZON\MailQueue\Dao\MailQueueDao;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\Utils\Date;
use Throwable;

class TaskMailQueueService {

    /**
     * @var Mailer
     */
    protected $mailer;

    /**
     * @var MailQueueDao
     */
    protected $mailQueueDao;

    /**
     * @var MailQueueAttachmentDao
     */
    protected $mailQueueAttachmentDao;

    /**
     * BaseMailQueueService constructor.
     * @param MailQueueDao $mailQueueDao
     * @param MailQueueAttachmentDao $mailQueueAttachmentDao
     */
    public function __construct(
        Mailer $mailer,
        MailQueueDao $mailQueueDao,
        MailQueueAttachmentDao $mailQueueAttachmentDao
    ) {
        $this->mailer = $mailer;
        $this->mailQueueDao = $mailQueueDao;
        $this->mailQueueAttachmentDao = $mailQueueAttachmentDao;
    }

    /**
     * Return id MailQueue object - mail to send
     * Return null - nothing to send
     * @return int|null
     */
    public function getNextMailIdToSend() {
        $mailQueue = $this->mailQueueDao->find(['status' => MailQueue::STATUS_NEW], ['id' => OrderBy::ASC])
            ->setMaxResults(1)->uniqueResult();
        return $mailQueue === null ? null : $mailQueue->getId();
    }

    /**
     * returns id of next MailQueue object
     * @param array $excluded
     * @return int|null
     */
    public function getNextFailedMailIdToSend(array $excluded, int $resendFailedYoungerThanMinutes) {
        $date = new Date();
        $date->subTime(0, $resendFailedYoungerThanMinutes);
        $params = ['date' => $date->format('Y-m-d H:i:s')];
        if(!empty($excluded)) {
            $params['excluded'] = $excluded;
        }

        $mailQueue = $this->mailQueueDao
            ->customFindNextFailedMailToSend($params)
            ->setMaxResults(1)
            ->uniqueResult();

        return $mailQueue === null ? null : $mailQueue->getId();
    }

    /**
     * @param int $queueMailId
     * @return MailQueue|null
     * @throws Exception
     */
    public function sendMail(int $queueMailId) {
        /** @var MailQueue $queueMail */
        $queueMail = $this->mailQueueDao->findLockedById($queueMailId)->uniqueResult();
        if(
            $queueMail === null
            || !in_array(
                $queueMail->getStatus(),
                [MailQueue::STATUS_NEW, MailQueue::STATUS_FAIL]
            ) // do not handle other than
        ) {
            return null;
        }

        $queueMail->setAttachments(
            $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $queueMail->getId()])->listResult()
        );

        try {
            $queueMail->setStatus(MailQueue::STATUS_SENDING);
            $this->mailQueueDao->update($queueMail);
            $mail = $this->createMailObject($queueMail);
            $this->mailer->sendMail($mail);
            $queueMail->setStatus(MailQueue::STATUS_SENDED);
            $this->mailQueueDao->update($queueMail);
        } catch(Throwable $exception) {
            $queueMail->setStatus(MailQueue::STATUS_FAIL);
            $queueMail->setStatusText($exception->getMessage());
            $this->mailQueueDao->update($queueMail);
        }
        return $queueMail;
    }


    /**
     * @param MailQueue $mailQueue
     * @return Mail
     * @throws MailQueueException
     */
    protected function createMailObject(MailQueue $mailQueue) {
        $mail = new Mail();
        $mail->setSubject($mailQueue->getSubject());
        $mail->setHtml($mailQueue->isHtml());
        $mail->setFrom($mailQueue->getFromMail(), $mailQueue->getFromName());
        $mail->setBody($mailQueue->getContent());
        foreach($mailQueue->getToMails() as $to){
            if(!is_array($to)) {
                continue;
            }
            $name = $to['name'];
            $email = $to['email'];
            $mail->addTo($email,$name);
        }
        if(!empty($mailQueue->getReplyToMails())) {
            foreach($mailQueue->getReplyToMails() as $to) {
                if(!is_array($to)) {
                    continue;
                }
                $name = $to['name'];
                $email = $to['email'];
                $mail->addReplyTo($email, $name);
            }
        }
        foreach($mailQueue->getAttachments() as $attachment) {
            if( $attachment->getCid() !== null ) {
                $mail->addEmbeddedImage(
                    $attachment->getAttachment()->getFsPath(),
                    $attachment->getCid(),
                    $attachment->getName(),
                    $attachment->getAttachment()->getMimeType()
                );
            } else {
                $mail->addAttachment(
                    $attachment->getAttachment()->getFsPath(),
                    $attachment->getName(),
                    $attachment->getAttachment()->getMimeType()
                );
            }
        }
        return $mail;
    }


    /**
     * get mail queue ides older than $date
     * @param Date $date
     * @return MailQueue[]
     */
    public function getMailIdesOlderThan(Date $date): array {
        $mails = $this->mailQueueDao->findMailsOlderThan($date->format('Y-m-d H:i:s'))->listResult(
            QueryResult::FETCH_ASSOC
        );
        return array_map(
            function($mailArray) {
                return $mailArray['id'];
            },
            $mails
        );
    }

    /**
     * @param int $mailId
     */
    public function deleteMail(int $mailId) {
        $mail = $this->mailQueueDao->find(['id' => $mailId])->uniqueResult();
        if( $mail === null ) {
            return;
        }

        $attachments = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $mailId])->listResult();
        foreach($attachments as $attachment) {
            $this->mailQueueAttachmentDao->delete($attachment);
        }
        $this->mailQueueDao->delete($mail);
    }
}
