<?php

namespace IZON\MailQueue\Task;


use IZON\Logs\Logger;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Task\Services\TaskMailQueueService;
use IZON\MailQueue\Web\Services\MailQueueService;
use IZON\Tasks\Task;
use IZON\Utils\Date;


class QueueMailSenderTask implements Task {

    /**
     * @var TaskMailQueueService
     */
    protected $mailQueueService;

    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var int Limit of sended mail per one run of task
     */
    protected $limitOfMailOnRun = 100;

    /**
     * @var int Num of days, half a year
     * after that num of day The mail will be deleted
     */
    protected $olderThanToDelete = 60;

    /**
     * @var bool Shall delete old emails
     */
    protected $deleteOldMails = true;

    /**
     * If true try to resend in failed state.
     * @var bool
     */
    protected $resendFailed = true;

    /**
     * Tries to resend failed emails younger than given minutes from NOW. Default is 30 mins.
     * @var int
     */
    protected $resendFailedYoungerThanMinutes = 30;


    public function __construct(TaskMailQueueService $mailQueueService) {
        $this->mailQueueService = $mailQueueService;
    }


    /**
     * @param Mailer $mailer
     */
    public function setMailer(Mailer $mailer) {
        $this->mailer = $mailer;
    }

    /**
     * @param MailQueueService $mailQueueService
     */
    public function setMailQueueService(TaskMailQueueService $mailQueueService) {
        $this->mailQueueService = $mailQueueService;
    }

    /**
     * @param int $limitOfMailOnRun
     */
    public function setLimitOfMailOnRun(int $limitOfMailOnRun) {
        $this->limitOfMailOnRun = $limitOfMailOnRun;
    }

    /**
     * Set after how much day the mail will be deleted
     * @param int|null $olderThanToDelete
     */
    public function setOlderThanToDelete(int $olderThanToDelete): void {
        $this->olderThanToDelete = $olderThanToDelete;
    }

    /**
     * Set if you want/do not wand delete old emails
     * @param bool $deleteOldMails
     */
    public function setDeleteOldMails(bool $deleteOldMails): void {
        $this->deleteOldMails = $deleteOldMails;
    }

        /**
     * @param bool $resendFailed
     */
    public function setResendFailed(bool $resendFailed): void {
        $this->resendFailed = $resendFailed;
    }

    /**
     * @param int $resendFailedYoungerThanMinutes
     */
    public function setResendFailedYoungerThanMinutes(int $resendFailedYoungerThanMinutes): void {
        $this->resendFailedYoungerThanMinutes = $resendFailedYoungerThanMinutes;
    }

    /**
     * @param array $args
     * @throws MailQueueException
     */
    public function run(array $args) {
        $failedTried = [];
        for($i=0; $i < $this->limitOfMailOnRun; $i++) {
            // find next email to send
            $queueMailId = $this->mailQueueService->getNextMailIdToSend();
            if( empty($queueMailId) && $this->resendFailed ) {
                $queueMailId = $this->mailQueueService->getNextFailedMailIdToSend(
                    $failedTried,
                    $this->resendFailedYoungerThanMinutes
                );
                if(!empty($queueMailId)) {
                    $failedTried[] = $queueMailId;
                }
            }
            if( empty($queueMailId) ) {
                break;
            }
            try {
                $this->mailQueueService->sendMail($queueMailId);
            } catch(\Throwable $exception) {
                throw $exception;
                $this->getLogger()->error('Sending of mail id = '. $queueMailId ." fail", ['exception' => $exception]);
            }
        }
        $this->deleteOldMails();
    }

    /**
     * Delete old records of mails
     */
    protected function deleteOldMails() {
        if( $this->deleteOldMails ) {
            $date = new Date();
            $date->subTime(0, 0, 0, $this->olderThanToDelete);

            $mailIdes = $this->mailQueueService->getMailIdesOlderThan($date);
            foreach($mailIdes as $mailId) {
                try {
                    $this->mailQueueService->deleteMail($mailId);
                } catch (\Exception $exception){
                    $this->getLogger()->error('Deletion of mail with id = '. $mailId .' faild', ['exception' => $exception]);
                }
            }
        }
    }

    /**
     * @return Logger
     */
    protected function getLogger() {
        if(empty($this->logger)){
            $this->logger = Logger::getLogger(self::class);
        }
        return $this->logger;
    }

}
