<?php

namespace IZON\MailQueue;


use IZON\IO\File;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Domain\MailQueueAttachment;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Services\BaseMailQueueService;


class QueueMailer implements Mailer {

    /**
     * @var BaseMailQueueService
     */
    protected $mailQueueService;

    /**
     * @var string
     */
    protected $tmpDir;

    /**
     * @var string[] temporary files to delete after MailQueue saved
     */
    protected $tempFilePathsToDelete = [];


    public function __construct(
        BaseMailQueueService $mailQueueService,
        string $tmpDir
    ) {
        $this->mailQueueService = $mailQueueService;
        $this->tmpDir = $tmpDir;
    }


    /**
     * @param Mail $mail
     * @throws Exceptions\MailQueueException
     */
    public function sendMail(Mail $mail) {
        $queue = $this->createQueueMail($mail);
        $queue->setStatus(MailQueue::STATUS_NEW);

        $this->mailQueueService->save($queue);

        foreach($this->tempFilePathsToDelete as $filePathToDelete) {
            unlink($filePathToDelete);
        }
    }

    /**
     * Create queue mail by mail
     * @param Mail $mail
     * @return MailQueue
     * @throws Exceptions\MailQueueException
     */
    protected function createQueueMail(Mail $mail) {
        $queue = new MailQueue();
        if(empty($mail->getFrom())){
            throw new MailQueueException('Sender of mail can not be empty');
        }
        $queue->setFromMail($mail->getFrom()['email'], (is_null($mail->getFrom()['name'])? '': $mail->getFrom()['name']));
        if(empty($mail->getSubject())){
            $queue->setSubject("");
        }else {
            $queue->setSubject($mail->getSubject());
        }
        if(empty($mail->getBody())){
            $queue->setContent("");
        }else {
            $queue->setContent($mail->getBody());
        }
        $queue->setHtml($mail->isHtml());
        foreach($mail->getReplyTo() as $to){
            $queue->addReplyToMails($to['email'], (is_null($to['name']) ? '' : $to['name']));
        }
        foreach($mail->getTo() as $to){
            $queue->addToMail($to['email'], (is_null($to['name']) ? '' : $to['name']));
        }

        foreach($mail->getAttachments() as $attachment){
            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setName($attachment["name"]);
            $queueAttachment->setAttachment(new File($attachment["path"]));
            $queue->addAttachment($queueAttachment);
        }

        foreach($mail->getEmbeddedImages() as $embededImage) {
            $queueAttachment = new MailQueueAttachment();
            $queueAttachment->setName($embededImage["name"]);
            $queueAttachment->setCid($embededImage["cid"]);

            $filePath = $embededImage["path"];
            if( $embededImage['type'] == 'content' ) {
                $filePath = $this->writeAttachmentContentToFile($embededImage['name'], $embededImage['content']);
                $this->tempFilePathsToDelete[] = $filePath;
            }
            $queueAttachment->setAttachment(new File($filePath));
            $queue->addAttachment($queueAttachment);
        }

        return $queue;
    }

    /**
     * @param string $fileName
     * @param $fileContent
     * @return string path where is file stored
     */
    protected function writeAttachmentContentToFile(string $fileName, $fileContent): string {
        if( !file_exists($this->tmpDir) ) {
            mkdir($this->tmpDir, 0777, true);
        }

        $filePath = $this->tmpDir ."/". time() .'-'. $fileName;
        file_put_contents($filePath, $fileContent);
        return $filePath;
    }
}