<?php

namespace IZON\MailQueue\Task;

use IZON\Logs\Logger;
use IZON\Mailer\Mail;
use IZON\Mailer\Mailer;
use IZON\MailQueue\Domain\MailQueue;
use IZON\MailQueue\Exceptions\MailQueueException;
use IZON\MailQueue\Task\Services\MailQueueService;
use IZON\Tasks\BaseTasks\DatabaseLoggedTask;
use IZON\Tasks\Services\ITaskWatchdogService;
use IZON\Utils\Date;


class QueueMailSenderTask extends DatabaseLoggedTask {

    /**
     * @var Mailer
     */
    protected $mailer;
    /**
     * @var MailQueueService
     */
    protected $service;
    /**
     * @var Logger
     */
    protected $logger;

    /**
     * @var int Limit of sended mail per one run of task
     */
    protected $limitOfMailOnRun = 100;
    /**
     * @var int Num of days,
     * after that num of day The mail will be deleted
     */
    protected $olderThanToDelete = 300;
    /**
     * @var bool Shall delete old emails
     */
    protected $deleteOldMails = true;

    public function __construct(Mailer $mailer, MailQueueService $mailQueueService, ITaskWatchdogService $taskWatchDogService) {
        parent::__construct($taskWatchDogService);
        $this->mailer = $mailer;
        $this->service = $mailQueueService;
    }


    /**
     * @param Mailer $mailer
     */
    public function setMailer(Mailer $mailer) {
        $this->mailer = $mailer;
    }

    /**
     * @param MailQueueService $service
     */
    public function setService(MailQueueService $service) {
        $this->service = $service;
    }

    /**
     * @param int $limitOfMailOnRun
     */
    public function setLimitOfMailOnRun(int $limitOfMailOnRun) {
        $this->limitOfMailOnRun = $limitOfMailOnRun;
    }

    /**
     * Set after how much day the mail will be deleted
     * @param int|null $olderThanToDelete
     */
    public function setOlderThanToDelete(int $olderThanToDelete): void {
        $this->olderThanToDelete = $olderThanToDelete;
    }

    /**
     * Set if you want/do not wand delete old emails
     * @param bool $deleteOldMails
     */
    public function setDeleteOldMails(bool $deleteOldMails): void {
        $this->deleteOldMails = $deleteOldMails;
    }



    /**
     * @param array $args
     * @throws MailQueueException
     */
    public function runTask(array $args) {
        //TODO: try to resend mails in faild state
        for($i=0; $i < $this->limitOfMailOnRun; $i++){
            try {
                $queueMail = $this->service->getNextMailToSend();
                if( empty($queueMail) ) {
                    break;
                }
            }catch(\Exception $e){
                $this->getLogger()->error('Some mail send fail', ['exception' => $e]);
                continue;
            }
            try {
                $queueMail->setStatus(MailQueue::STATUS_SENDING);
                $this->service->update($queueMail);
                $mail = $this->createMailObject($queueMail);
                $this->mailer->sendMail($mail);
                $queueMail->setStatus(MailQueue::STATUS_SENDED);
                $this->service->update($queueMail);
            }catch(\Exception $exception){
                $queueMail->setStatus(MailQueue::STATUS_FAIL);
                $queueMail->setStatusText($exception->getMessage());
                $this->service->update($queueMail);
                $this->getLogger()->error('Send mail id:' . $queueMail->getId() . " fail", ['exception' => $exception]);
                continue;
            }
        }
        $this->deleteOldRecords();
    }

    /**
     * Delete old records of mails
     */
    protected function deleteOldRecords(){
        if($this->deleteOldMails) {
            try {
                $date = new Date();
                $date->subTime(0, 0, 0, $this->olderThanToDelete);
                $mails = $this->service->getOlderMailsThanInState($date, MailQueue::STATUS_SENDED);
                $this->service->deleteMails($mails);
            } catch(\Exception $exception){
                $this->getLogger()->error('Delete old mails faild', ['exception' => $exception]);
            }
        }
    }

    /**
     * @param MailQueue $mailQueue
     * @return Mail
     * @throws MailQueueException
     */
    protected function createMailObject(MailQueue $mailQueue) {
        $mail = new Mail();
        $mail->setSubject($mailQueue->getSubject());
        $mail->setHtml($mailQueue->isHtml());
        $mail->setFrom($mailQueue->getFromMail(), $mailQueue->getFromName());
        $mail->setBody($mailQueue->getContent());
        foreach($mailQueue->getToMails() as $to){
            if(!is_array($to)) {
                continue;
            }
            $name = $to['name'];
            $email = $to['email'];
            $mail->addTo($email,$name);
        }
        if(!empty($mailQueue->getReplyToMails())) {
            foreach($mailQueue->getReplyToMails() as $to) {
                if(!is_array($to)) {
                    continue;
                }
                $name = $to['name'];
                $email = $to['email'];
                $mail->addReplyTo($email, $name);
            }
        }
        foreach($mailQueue->getAttachments() as $attachment){
            $mail->addAttachment($attachment->getPath(),$attachment->getName(), $attachment->getType(), $attachment->getEncoding());
        }
        return $mail;
    }

    /**
     * @return Logger
     */
    protected function getLogger(){
        if(empty($this->logger)){
            $this->logger = Logger::getLogger(self::class);
        }
        return $this->logger;
    }


}