<?php

namespace IZON\MailQueue\Services;

use IZON\DB\DBConnection;
use IZON\MailQueue\Dao\MailQueueAttachmentDao;
use IZON\MailQueue\Dao\MailQueueDao;
use IZON\MailQueue\Domain\MailQueue;
use IZON\Utils\Date;


class BaseMailQueueService {

    /**
     * @var DBConnection
     */
    protected $dbConnection;
    
    /**
     * @var MailQueueDao
     */
    protected $mailQueueDao;
    
    /**
     * @var MailQueueAttachmentDao
     */
    protected $mailQueueAttachmentDao;

    /**
     * BaseMailQueueService constructor.
     * @param MailQueueDao $mailQueueDao
     * @param MailQueueAttachmentDao $mailQueueAttachmentDao
     */
    public function __construct(MailQueueDao $mailQueueDao, MailQueueAttachmentDao $mailQueueAttachmentDao) {
        $this->mailQueueDao = $mailQueueDao;
        $this->mailQueueAttachmentDao = $mailQueueAttachmentDao;
    }

    /**
     * @param MailQueue $mailQueue
     */
    protected function fill(MailQueue $mailQueue){
        $attachements = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $mailQueue->getId()])->listResult();
        $mailQueue->setAttachments($attachements);
    }

    /**
     * @param $id
     * @return MailQueue
     */
    public function get($id){
        return $this->executeTransactional(function() use($id) {
            $mailQueue = $this->mailQueueDao->load($id);
            $this->fill($mailQueue);
            return $mailQueue;
        });
    }

    /**
     * @return MailQueue[]
     */
    public function getAllNewInQueue() {
        return $this->executeTransactional(function() {
            $mailQueue = $this->mailQueueDao->find(['status' => MailQueue::STATUS_NEW])->listResult();
            foreach($mailQueue as $item) {
                $this->fill($item);
            }
            return $mailQueue;
        });
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function save(MailQueue $mailQueue){
        $this->executeTransactional(function() use ($mailQueue) {
            $mailQueue->setDate(new Date());
            $this->mailQueueDao->save($mailQueue);
            foreach($mailQueue->getAttachments() as $item){
                $item->setFkMailQueueId($mailQueue->getId());
                $this->mailQueueAttachmentDao->save($item);
            }
        });
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function update(MailQueue $mailQueue){
        $this->executeTransactional(function() use ($mailQueue) {
            $this->mailQueueDao->update($mailQueue);
            $this->updateAttachmentObjects($mailQueue);
        });
    }

    /**
     * @param MailQueue $mailQueue
     */
    public function delete(MailQueue $mailQueue){
        $this->executeTransactional(function() use ($mailQueue) {
            $mailQueue->setAttachments([]);
            $this->updateAttachmentObjects($mailQueue);
            $this->mailQueueDao->delete($mailQueue);
        });
    }

    /**
     * To update attachment - without transation
     * @param MailQueue $mailQueue
     */
    protected function updateAttachmentObjects(MailQueue $mailQueue) {
        $olds = $this->mailQueueAttachmentDao->find(['fkMailQueueId' => $mailQueue->getId()])->listResult();
        $oldsHash = [];
        foreach($olds as $old){
            $oldsHash[$old->getId()] = $old;
        }
        foreach($mailQueue->getAttachments() as $object){
            if(isset($oldsHash[$object->getId()])){
                unset($oldsHash[$object->getId()]);
            }else{
                $object->setFkMailQueueId($mailQueue->getId());
                $this->mailQueueAttachmentDao->save($object);
            }
        }
        foreach($oldsHash as $toDelete){
            $this->mailQueueAttachmentDao->delete($toDelete);
        }
    }

    public function setDBConnection(DBConnection $dbConnection) {
        $this->dbConnection = $dbConnection;
    }
    
    /**
     * executes $transactional callable in transaction (creates new transaction if transaction not opened yet)
     * @param callable $transactional
     * @return mixed value returned from transactional callable
     */
    protected function executeTransactional(callable $transactional) {
        $retval = null;
        if( $this->dbConnection !== null ) { // we have connection to start transaction on and test if transaction exists
            $transactionStarted = false;
            if( !$this->dbConnection->inTransaction() ) { // transaction not opened yet
                $this->dbConnection->beginTransaction(); // start transaction
                $transactionStarted = true;
            }

            // execcute transactional
            $retval = $transactional();

            if( $transactionStarted ) {
                $this->dbConnection->commit(); // commit started transaction
            }
        } else {
            // process transactions as usual
            $this->mailQueueDao->beginTransaction();
            
            // execute transactional
            $retval = $transactional();
            
            $this->mailQueueDao->commit();
        }
        
        
        return $retval;
    }
}