<?php

namespace IZON\Logs\Tests;

use IZON\Logs\Exceptions\LoggerException;
use IZON\Logs\LoggerInitializer;
use IZON\Logs\LoggerRegistry;
use Monolog\Handler\ErrorLogHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

class LoggerInitializerTest extends TestCase
{
    protected function setUp(): void
    {
        LoggerRegistry::clearRegistry();
    }

    /**
     * initializes from default file logger.config.php in current working directory
     */
    public function testNoDefaultConfigFile(): void
    {
        chdir(dirname(__DIR__)); // set current working directory to the directory of this test file

        $this->expectException(LoggerException::class);

        LoggerInitializer::init();
    }

    /**
     * initializes from default file logger.config.php in current working directory
     */
    public function testInitializeWithDefaultConfigFile(): void
    {
        chdir(__DIR__); // set current working directory to the directory of this test file

        LoggerInitializer::init();

        $logger = LoggerRegistry::getLogger(LoggerRegistry::DEFAULT_CHANNEL);
        $this->assertInstanceOf(LoggerInterface::class, $logger); // @phpstan-ignore method.alreadyNarrowedType
        $this->assertInstanceOf(Logger::class, $logger);
        $this->assertCount(1, $logger->getHandlers());
        $this->assertInstanceOf(StreamHandler::class, $logger->getHandlers()[0]);
    }

    /**
     * chovani pro clearRegistry
     */
    public function testUninitializedAfterClear(): void
    {
        chdir(__DIR__); // set current working directory to the directory of this test file

        LoggerInitializer::init();

        LoggerRegistry::clearRegistry();

        // ma vratit defautni registry
        $logger = LoggerRegistry::getLogger(LoggerRegistry::DEFAULT_CHANNEL);
        $this->assertInstanceOf(LoggerInterface::class, $logger); // @phpstan-ignore method.alreadyNarrowedType
        $this->assertInstanceOf(Logger::class, $logger);
        $this->assertCount(1, $logger->getHandlers());
        $this->assertInstanceOf(ErrorLogHandler::class, $logger->getHandlers()[0]);
    }

    /**
     * testuje pokud je v environmentu nastaven název konfigurace která neexistuje
     */
    public function testInitWithNonExistentConfigFile(): void
    {
        LoggerInitializer::setConfigFilePath(__DIR__ . '/nonexistant.config.php');

        $this->expectException(LoggerException::class);
        LoggerInitializer::init();
    }

    /**
     *  tests nonexistent config file override in env
     */
    public function testInitWithNonExistentConfigFileEnv(): void
    {
        LoggerRegistry::clearRegistry();
        putenv(LoggerInitializer::DEFAULT_CONFIG_PATH_ENV_VAR . '=' . __DIR__ . '/nonexistant.config.php');

        $this->expectException(LoggerException::class);
        LoggerInitializer::init();
    }

    /**
     *  tests empty config file
     */
    public function testInitWithEmptyFile(): void
    {
        LoggerInitializer::setConfigFilePath(__DIR__ . '/log-configs/empty.php');

        $this->expectException(LoggerException::class);
        LoggerInitializer::init();
    }

    /**
     * testuje vytáhnutí loggeru bes defaultního kanálu
     */
    public function testConfigWithoutDefault(): void
    {
        LoggerInitializer::setConfigFilePath(__DIR__ . '/log-configs/no-default-defined.php');

        $this->expectException(LoggerException::class);
        LoggerInitializer::init();
    }

    public function testConfigWithoutValidLogger(): void
    {
        LoggerInitializer::setConfigFilePath(__DIR__ . '/log-configs/no-default-defined.php');

        $this->expectException(LoggerException::class);
        LoggerInitializer::init();
    }

    public function testConfigWithoutMatchingKeyAndLoggerName(): void
    {
        LoggerInitializer::setConfigFilePath(__DIR__ . '/log-configs/not-matching-key.php');

        $this->expectException(LoggerException::class);
        LoggerInitializer::init();
    }
}
