<?php

namespace IZON\Logs;

use IZON\Logs\Exceptions\LoggerException;
use Monolog\Logger;
use Psr\Log\LoggerInterface;

class LoggerInitializer
{
    /**
     * environment variable name to inspect for the logger configuration file path
     */
    public const DEFAULT_CONFIG_PATH_ENV_VAR = 'IZON_LOGGER_CONFIG_FILE';

    /**
     * default configuration file name searched in current working directory
     */
    public const DEFAULT_CONFIG_FILE_NAME = 'logger.config.php';

    /**
     * configuration file path to use if DEFAULT_CONFIG_PATH_ENV_VAR is not set
     */
    protected static string $configFilePath = self::DEFAULT_CONFIG_FILE_NAME;

    /**
     * @param string $configFilePath path to the logger configuration file
     */
    public static function setConfigFilePath(string $configFilePath): void
    {
        self::$configFilePath = $configFilePath;
    }


    /**
     * function to initialize the logger
     * - check if DEFAULT_CONFIG_PATH_ENV_VARIABLE is set
     * - if env var is set take config file path from it. If not take path set by call of setConfigFilePath of default value of DEFAULT_CONFIG_FILE_NAME
     * - check if config file exists in specified path (If relative path is used, it searches relative to current working directory )
     * - throw LoggerException if file does not exist
     * - throw LoggerException if file does not contain an array of loggers
     * - throw LoggerException if array of loggers does not contain default logger array key
     * - init LoggerRegistry with loggers from configuration file
     *
     * Loggers are than available in LoggerRegistry
     *
     * @return void
     */
    public static function init(): void
    {
        $configFile = getenv(self::DEFAULT_CONFIG_PATH_ENV_VAR) ?: self::$configFilePath;

        if (!file_exists($configFile)) {
            throw new LoggerException(
                sprintf('The logger configuration file is missing in path [%s].', $configFile)
            );
        }

        $loggers = require $configFile;

        if (!is_array($loggers)) {
            throw new LoggerException(
                sprintf('The logger configuration file [%s] does not return an array of loggers.', $configFile)
            );
        }

        foreach ($loggers as $channelName => $logger) {
            if (!$logger instanceof LoggerInterface) {
                throw new LoggerException(
                    sprintf('Key [%s] does not contain valid logger.', $channelName)
                );
            }
            if (!$logger instanceof Logger) {
                throw new LoggerException(
                    sprintf('Key [%s] does not contain valid Monolog logger.', $channelName)
                );
            }
            if ($channelName != $logger->getName()) {
                throw new LoggerException(
                    sprintf('Key [%s] does not match logger name [%s].', $channelName, $logger->getName())
                );
            }
        }

        if (!array_key_exists(LoggerRegistry::DEFAULT_CHANNEL, $loggers)) {
            throw new LoggerException(sprintf('The default logger configuration is not defined in file [%s].', $configFile));
        }

        LoggerRegistry::clearRegistry();

        foreach ($loggers as $channel => $logger) {
            LoggerRegistry::addLogger($channel, $logger);
        }
    }
}
