<?php

namespace IZON\Logs;

use IZON\Logs\Handlers\EchoHandler;
use Monolog\Logger as MonologLogger;

/**
 * factory to create loggers from passed configuration
 */
class LoggerFactory implements LoggerFactoryInterface, ConfigurableInterface
{
    /**
     *
     * @var self
     */
    protected static ?LoggerFactory $singleton = null;
    protected ?MonologLogger $defaultLogger = null;
    protected ?MonologLogger $configuredLogger = null;
    protected bool $isConfigured = false;
    /**
     *
     * @var array<string, Logger>
     */
    protected array $loggers = [];
    /**
     *
     * @var DataAppenderInterface[]
     */
    protected array $appenders = [];

    final public function __construct(?MonologLogger $defaultLogger = null)
    {
        if (empty($defaultLogger)) {
            $defaultLogger = self::createDefaultLogger();
        }
        $this->defaultLogger = $defaultLogger;
        if (empty(self::$singleton)) {
            self::$singleton = $this;
        }
    }

    /**
     * create default logger
     *
     * @return MonologLogger
     */
    protected static function createDefaultLogger()
    {
        $logger = new MonologLogger('default');
        $logger->pushHandler(new EchoHandler());
        return $logger;
    }

    /**
     *
     * @param string|null $name
     * @return Logger
     */
    public static function getLogger(?string $name): Logger
    {
        if (empty(self::$singleton)) {
            self::$singleton = new static();
        }
        return self::$singleton->createLogger($name);
    }

    /**
     * create logger
     *
     * @param string|null $name
     * @return Logger
     */
    public function createLogger(?string $name): Logger
    {
        if (!$this->isConfigured) {
            $logger = $this->defaultLogger;
        } else {
            $logger = $this->configuredLogger;
        }
        if (array_key_exists($name, $this->loggers)) {
            return $this->loggers[$name];
        }
        $newLogger = new Logger($logger->withName($name), $this->appenders);
        $this->loggers[$name] = $newLogger;
        return $newLogger;
    }

    /**
     * configure factory
     * @param array $configuration
     */
    public static function configure(array $configuration)
    {
        if (empty(self::$singleton)) {
            self::$singleton = new static();
        }
        self::$singleton->configureFactory($configuration);
    }

    /**
     * configure instance of factory
     *
     * @param array $config
     * @return void
     */
    public function configureFactory(array $config)
    {
        if ($this->isConfigured) {
            return;
        }
        $handlers = $config['handlers'];
        if (array_key_exists('channels', $config)) {
            $channels = $config['channels'];
        }
        if (array_key_exists('appenders', $config)) {
            $this->appenders = $config['appenders'];
        }
        if (empty($handlers)) {
            return;
        }
        $defaultHandlers = $handlers;

        if (!empty($channels) && array_key_exists('default', $channels)) {
            $handlerNames = $channels['default'];
            $defaultHandlers = [];
            foreach ($handlers as $key => $handler) {
                if (!in_array($key, $handlerNames)) {
                    continue;
                }
                $defaultHandlers[$key] = $handler;
            }
        }

        $logger = new MonologLogger('default');
        foreach ($defaultHandlers as $handler) {
            $logger->pushHandler($handler);
        }
        $this->configuredLogger = $logger;
        $this->isConfigured = true;

        if (empty($channels)) {
            return;
        }
        foreach ($channels as $channel => $channelHandlers) {
            if ($channel == 'default') {
                continue;
            }
            $mLogger = new MonologLogger($channel);
            foreach ($channelHandlers as $handlerName) {
                if (!array_key_exists($handlerName, $handlers)) {
                    return;
                }
                $mLogger->pushHandler($handlers[$handlerName]);
            }
            $this->loggers[$channel] = new Logger($mLogger, $this->appenders);
        }
    }
}
