<?php


namespace IZON\Logs;

use IZON\Logs\Handlers\EchoHandler;
use Monolog\Logger as MonologLogger;

/**
 * Description of LoggerFactory
 *
 * @author lukas
 */
class LoggerFactory implements LoggerFactoryInterface, ConfigurableInterface {
    /**
     *
     * @var MonologLogger
     */
    protected $defaultLogger = null;
    /**
     *
     * @var MonologLogger
     */
    protected $configuredLogger = null;
    /**
     *
     * @var bool
     */
    protected $isConfigured = false;
    /**
     *
     * @var self
     */
    protected static $singleton;
    /**
     *
     * @var logger
     */
    protected $loggers = [];
    /**
     *
     * @var DataAppenderInterface[]
     */
    protected $appenders = [];

    function __construct(MonologLogger $defaultLogger = null) {
        if(empty($defaultLogger)) {
            $defaultLogger = self::createDefaultLogger();
        }
        $this->defaultLogger = $defaultLogger;
        if(empty(self::$singleton)) {
            self::$singleton = $this;
        }
    }

    /**
     * create default logger
     * 
     * @return MonologLogger
     */
    protected static function createDefaultLogger() {
        $logger = new MonologLogger('default');
        $logger->pushHandler(new EchoHandler());
        return $logger;
    }

    /**
     * 
     * @param string|null $name
     * @return Logger
     */
    public static function getLogger(?string $name): Logger {
        if(empty(self::$singleton)) {
            self::$singleton = new static();
        }
        return self::$singleton->createLogger($name);
    }

    /**
     * configure factory
     * @param array $configuration
     */
    public static function configure(array $configuration) {
        if(empty(self::$singleton)) {
            self::$singleton = new static();
        }
        self::$singleton->configureFactory($configuration);
    }

    /**
     * configure instance of factory
     * 
     * @param array $config
     * @return void
     */
    public function configureFactory(array $config) {
        $handlers = $config['handlers'];
        if(array_key_exists('appenders', $config)) {
            $this->appenders = $config['appenders'];
        }
        if(empty($handlers)) {
            return;
        }
        $logger = new MonologLogger('default');
        foreach($handlers as $handler) {
            $logger->pushHandler($handler);
        }
        $this->configuredLogger = $logger;
        $this->isConfigured = true;
    }

    /**
     * create logger
     * 
     * @param string|null $name
     * @return Logger
     */
    public function createLogger(?string $name): Logger {
        if(!$this->isConfigured) {
            $logger = $this->defaultLogger;
        } else {
            $logger = $this->configuredLogger;
        }
        if(array_key_exists($name, $this->loggers)) {
            return $this->loggers[$name];
        }
        $newLogger = new Logger($logger->withName($name), $this->appenders);
        $this->loggers[$name] = $newLogger;
        return $newLogger;
    }

}
