<?php

namespace IZON\Logs\Handlers;

use IZON\Logs\Exceptions\LoggerException;
use Monolog\Handler\StreamHandler;
use Monolog\Logger;
use function IZON\File\move;

/**
 * echo log to standard output
 *
 * @author lukas
 */
class RotatingFileSizeHandler extends StreamHandler {
    const MAX_FILE_SIZE = '5MB';
    protected $filename;
    protected $maxFiles;
    protected $maxFileSize;
    protected $mustRotate;
    protected $filenameFormat;
    protected $dateFormat;

    /**
     * @param string     $filename
     * @param int        $maxFiles       The maximal amount of files to keep (0 means unlimited)
     * @param string     $maxFileSize
     * @param string|int $level          The minimum logging level at which this handler will be triggered
     * @param bool       $bubble         Whether the messages that are handled can bubble up the stack or not
     * @param int|null   $filePermission Optional file permissions (default (0644) are only for owner read/write)
     * @param bool       $useLocking     Try to lock log file before doing any writes
     */
    public function __construct(
        string $filename,
        int $maxFiles = 0,
        string $maxFileSize = self::MAX_FILE_SIZE,
        $level = Logger::DEBUG,
        bool $bubble = true,
        ?int $filePermission = null,
        bool $useLocking = false
    ) {
        $this->filename = $filename;
        $this->maxFiles = $maxFiles;

        $this->maxFileSize = $this->convertToBytes($maxFileSize);
        
        $this->filenameFormat = '{filename}';
        parent::__construct($filename, $level, $bubble, $filePermission, $useLocking);
    }

    /**
     * {@inheritdoc}
     */
    public function close(): void {
        parent::close();
        if(true === $this->mustRotate) {
            $this->rotate();
        }
    }

    /**
     * {@inheritdoc}
     */
    public function reset() {
        parent::reset();
        if(true === $this->mustRotate) {
            $this->rotate();
        }
    }

    /**
     * {@inheritdoc}
     */
    protected function write(array $record): void {
        if(file_exists($this->filename) && filesize($this->filename) >= $this->maxFileSize) {
            $this->mustRotate = true;
            $this->close();
        }
        parent::write($record);
    }

    /**
     * Rotates the files.
     */
    protected function rotate(): void {
        // skip GC of old logs if files are unlimited
        if(0 === $this->maxFiles) {
            return;
        }
        $oldestFileIndex = $this->maxFiles - 1;
        if($oldestFileIndex === 0) {
            $fp = fopen($this->filename, "w");
            fclose($fp);
            $this->mustRotate = false;
            return;
        }
        $oldestFile = $this->filename.".". $oldestFileIndex;
        // Make Windows happy and delete file :D
        if (file_exists($oldestFile) && !unlink($oldestFile)) {
				throw new LoggerException("Unable to delete oldest backup file from [$oldestFile].");
        }
        for($currentIndex = $oldestFile; $currentIndex > 0; $currentIndex--) {
            $nextIndex = $currentIndex - 1;
            if($nextIndex == 0) {
                $sourceFileName = $this->filename;
            } else {
                $sourceFileName = $this->filename.".".$nextIndex;
            }
            $targetFileName = $this->filename.".".$currentIndex;
            move($sourceFileName, $targetFileName);
        }
        $this->mustRotate = false;
    }

    protected function getGlobPattern(): string {
        return $this->filename."*";
    }

    protected function convertToBytes(string $from): ?int {
        $units = ['B', 'KB', 'MB', 'GB', 'TB', 'PB'];
        $number = substr($from, 0, -2);
        $suffix = strtoupper(substr($from, -2));

        //B or no suffix
        if(is_numeric(substr($suffix, 0, 1))) {
            return preg_replace('/[^\d]/', '', $from);
        }

        $exponent = array_flip($units)[$suffix] ?? null;
        if($exponent === null) {
            return null;
        }

        return $number * (1024 ** $exponent);
    }

}
