<?php

namespace IZON\Logs;

use Exception;
use \Psr\Log\LoggerInterface;
use \Logger as Log4PHPLogger;

/**
 * trida slouzici k logovani
 * 
 * rozhrani je zalozeno na http://www.php-fig.org/psr/psr-3/
 * zalozeno na Apache Log4php https://logging.apache.org/log4php/
 * 
 */
class Logger implements LoggerInterface {
    
    /**
     * @var Log4PHPLogger
     */
    protected $logger = NULL;
    
    protected function __construct($logger) {
        $this->logger = $logger;
    }
    
    /**
     * vrati loger
     * @param string $name nazev logeru
     * @return Logger
     */
    public static function getLogger($name) {
//        if( $this->logger === NULL) {
//            $this->logger = new Logger(Log4PHPLogger::getLogger($name));
//        }
//        return $this->logger;
        
        return new Logger(ExtendedLog4PHPLogger::getLogger($name));
    }
    
    public static function configure($configuration = null, $configurator = null) {
        ExtendedLog4PHPLogger::configure($configuration, $configurator);
    }
    
    public function emergency($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->fatal($message, $exception);
    }
    
    public function alert($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->error($message, $exception);
    }

    public function critical($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->fatal($message, $exception);
    }

    /**
     * 
     * @param int $level currntly supported levels from LogLevel
     * @param type $message
     * @param array $context
     */
    public function log($level, $message, array $context = []) {
        $level = \LoggerLevel::toLevel($level);
        $exception = $this->getExceptionFromContext($context);
        $this->logger->log($level, $message, $exception);
    }

    public function notice($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->info($message, $exception);
    }

    public function warning($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->warn($message, $exception);
    }

    public function debug($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->debug($message, $exception);
    }

    public function error($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->error($message, $exception);
    }

    public function info($message, array $context = []) {
        $exception = $this->getExceptionFromContext($context);
        $this->logger->info($message, $exception);
    }

    /**
     * @param array $context
     * @return Exception|null
     */
    protected function getExceptionFromContext(array $context) {
        if( isset($context['exception']) ) {
            $throwable = $context['exception'];
            if( $throwable instanceof \Throwable ) {
                return new \Exception($throwable->getMessage(), $throwable->getCode(), $throwable->getPrevious());
            }
            return $throwable;
        }
        
        return null;
    }
}

///////////////////////////////////////////////////
/**
 * class used to adjust file and line log
 */
class ExtendedLog4PHPLogger extends Log4PHPLogger {
    // ******************************************
    // *** Static methods and properties      ***
    // ******************************************

    /** The logger hierarchy used by log4php. */
    private static $hierarchy;

    /** Inidicates if log4php has been initialized */
    private static $initialized = false;

    /** 
     * The Logger's fully qualified class name.
     * TODO: Determine if this is useful. 
     */
    private $fqcn = ExtendedLog4PHPLogger::class;
    
    /**
     * Returns the hierarchy used by this Logger.
     *
     * Caution: do not use this hierarchy unless you have called initialize().
     * To get Loggers, use the Logger::getLogger and Logger::getRootLogger
     * methods instead of operating on on the hierarchy directly.
     *
     * @return LoggerHierarchy
     */
    public static function getHierarchy() {
        if (!isset(self::$hierarchy)) {
            self::$hierarchy = new ExtendedLoggerHierarchy(new \LoggerRoot());
        }
        return self::$hierarchy;
    }

    /**
     * Returns a Logger by name. If it does not exist, it will be created.
     *
     * @param string $name The logger name
     * @return ExtendedLog4PHPLogger
     */
    public static function getLogger($name) {
        if (!self::isInitialized()) {
            self::configure();
        }
        return self::getHierarchy()->getLogger($name);
    }

    /**
     * Returns the Root Logger.
     * @return LoggerRoot
     */
    public static function getRootLogger() {
        if (!self::isInitialized()) {
            self::configure();
        }
        return self::getHierarchy()->getRootLogger();
    }

    /**
     * Clears all Logger definitions from the logger hierarchy.
     * @return boolean
     */
    public static function clear() {
        return self::getHierarchy()->clear();
    }

    /**
     * Destroy configurations for logger definitions
     */
    public static function resetConfiguration() {
        self::getHierarchy()->resetConfiguration();
        self::getHierarchy()->clear(); // TODO: clear or not?
        self::$initialized = false;
    }

    /**
     * Safely close all appenders.
     * @deprecated This is no longer necessary due the appenders shutdown via
     * destructors.
     */
    public static function shutdown() {
        return self::getHierarchy()->shutdown();
    }

    /**
     * check if a given logger exists.
     *
     * @param string $name logger name
     * @return boolean
     */
    public static function exists($name) {
        return self::getHierarchy()->exists($name);
    }

    /**
     * Returns an array this whole Logger instances.
     * @see Logger
     * @return array
     */
    public static function getCurrentLoggers() {
        return self::getHierarchy()->getCurrentLoggers();
    }

    /**
     * Configures log4php.
     * 
     * This method needs to be called before the first logging event has 
     * occured. If this method is not called before then the default
     * configuration will be used.
     *
     * @param string|array $configuration Either a path to the configuration
     *   file, or a configuration array.
     *   
     * @param string|LoggerConfigurator $configurator A custom 
     * configurator class: either a class name (string), or an object which 
     * implements the LoggerConfigurator interface. If left empty, the default
     * configurator implementation will be used. 
     */
    public static function configure($configuration = null, $configurator = null) {
        self::resetConfiguration();
        $configurator = self::getConfigurator($configurator);
        $configurator->configure(self::getHierarchy(), $configuration);
        self::$initialized = true;
    }

    /**
     * Creates a logger configurator instance based on the provided 
     * configurator class. If no class is given, returns an instance of
     * the default configurator.
     * 
     * @param string|LoggerConfigurator $configurator The configurator class 
     * or LoggerConfigurator instance.
     */
    private static function getConfigurator($configurator = null) {
        if ($configurator === null) {
            return new \LoggerConfiguratorDefault();
        }

        if (is_object($configurator)) {
            if ($configurator instanceof LoggerConfigurator) {
                return $configurator;
            } else {
                trigger_error("log4php: Given configurator object [$configurator] does not implement the LoggerConfigurator interface. Reverting to default configurator.", E_USER_WARNING);
                return new \LoggerConfiguratorDefault();
            }
        }

        if (is_string($configurator)) {
            if (!class_exists($configurator)) {
                trigger_error("log4php: Specified configurator class [$configurator] does not exist. Reverting to default configurator.", E_USER_WARNING);
                return new \LoggerConfiguratorDefault();
            }

            $instance = new $configurator();

            if (!($instance instanceof LoggerConfigurator)) {
                trigger_error("log4php: Specified configurator class [$configurator] does not implement the LoggerConfigurator interface. Reverting to default configurator.", E_USER_WARNING);
                return new \LoggerConfiguratorDefault();
            }

            return $instance;
        }

        trigger_error("log4php: Invalid configurator specified. Expected either a string or a LoggerConfigurator instance. Reverting to default configurator.", E_USER_WARNING);
        return new \LoggerConfiguratorDefault();
    }

    /**
     * Returns true if the log4php framework has been initialized.
     * @return boolean 
     */
    private static function isInitialized() {
        return self::$initialized;
    }

    public function log(\LoggerLevel $level, $message, $throwable = null) {
        if ($this->isEnabledFor($level)) {
            $event = new ExtendedLoggerLoggingEvent($this->fqcn, $this, $level, $message, null, $throwable);
            $this->callAppenders($event);
        }

        // Forward the event upstream if additivity is turned on
        if ($this->getParent() !== NULL && $this->getAdditivity()) {

            // Use the event if already created
            if (isset($event)) {
                $this->getParent()->logEvent($event);
            } else {
                $this->getParent()->log($level, $message, $throwable);
            }
        }
    }

}

class ExtendedLoggerLoggingEvent extends \LoggerLoggingEvent {

    /** 
    * @var string Fully Qualified Class Name of the calling category class.
    */
    private $fqcn;

    /**
    * @var Logger reference
    */
    private $logger;

    /** 
     * The category (logger) name.
     * This field will be marked as private in future
     * releases. Please do not access it directly. 
     * Use the {@link getLoggerName()} method instead.
     * @deprecated 
     */
    private $categoryName;

    /** 
     * @var mixed The application supplied message of logging event. 
     */
    private $message;

    /** 
    * @var LoggerLocationInfo Location information for the caller. 
    */
    private $locationInfo;

    /**
     * @var LoggerThrowableInformation log4php internal representation of throwable
     */
    private $throwableInfo;

    /**
    * Instantiate a LoggingEvent from the supplied parameters.
    *
    * Except {@link $timeStamp} all the other fields of
    * LoggerLoggingEvent are filled when actually needed.
    *
    * @param string $fqcn name of the caller class.
    * @param mixed $logger The {@link Logger} category of this event or the logger name.
    * @param LoggerLevel $level The level of this event.
    * @param mixed $message The message of this event.
    * @param integer $timeStamp the timestamp of this logging event.
    * @param Exception $throwable The throwable associated with logging event
    */
    public function __construct($fqcn, $logger, \LoggerLevel $level, $message, $timeStamp = null, $throwable = null) {
            $this->fqcn = $fqcn;
            if($logger instanceof Log4PHPLogger) {
                    $this->logger = $logger;
                    $this->categoryName = $logger->getName();
            } else {
                    $this->categoryName = strval($logger);
            }
            $this->level = $level;
            $this->message = $message;
            if($timeStamp !== null && is_numeric($timeStamp)) {
                    $this->timeStamp = $timeStamp;
            } else {
                    $this->timeStamp = microtime(true);
            }

            if( $throwable !== null 
                && ($throwable instanceof Exception || $throwable instanceof \Throwable) ) {
                    $this->throwableInfo = new \LoggerThrowableInformation($throwable);
            }
            parent::__construct($fqcn, $logger, $level, $message, $timeStamp, $throwable);
    }

    /**
     * Set the location information for this logging event. The collected
     * information is cached for future use.
     *
     * <p>This method uses {@link PHP_MANUAL#debug_backtrace debug_backtrace()} function (if exists)
     * to collect informations about caller.</p>
     * <p>It only recognize informations generated by {@link Logger} and its subclasses.</p>
     * @return LoggerLocationInfo
     */
    public function getLocationInformation() {
        if ($this->locationInfo === null) {

            $locationInfo = array();
            $trace = debug_backtrace();
            $prevHop = null;
            // make a downsearch to identify the caller
            $hop = array_pop($trace);
            while ($hop !== null) {
                if (isset($hop['class'])) {
                    // we are sometimes in functions = no class available: avoid php warning here
                    $className = strtolower($hop['class']);
                    if (!empty($className)
                            and ( $className == strtolower(Logger::class)
                            or strtolower(get_parent_class($className)) == strtolower(Logger::class))) {
                        $locationInfo['line'] = $hop['line'];
                        $locationInfo['file'] = $hop['file'];
                        break;
                    }
                }
                $prevHop = $hop;
                $hop = array_pop($trace);
            }
            $locationInfo['class'] = isset($prevHop['class']) ? $prevHop['class'] : 'main';
            if (isset($prevHop['function']) and
                    $prevHop['function'] !== 'include' and
                    $prevHop['function'] !== 'include_once' and
                    $prevHop['function'] !== 'require' and
                    $prevHop['function'] !== 'require_once') {

                $locationInfo['function'] = $prevHop['function'];
            } else {
                $locationInfo['function'] = 'main';
            }

            $this->locationInfo = new \LoggerLocationInfo($locationInfo, $this->fqcn);
        }
        return $this->locationInfo;
    }

}

class ExtendedLoggerHierarchy extends \LoggerHierarchy {

    /**
     * Returns a named logger instance logger. If it doesn't exist, one is created.
     * 
     * @param string $name Logger name
     * @return Logger Logger instance.
     */
    public function getLogger($name) {
        if (!isset($this->loggers[$name])) {
            $logger = new ExtendedLog4PHPLogger($name);

            $nodes = explode('.', $name);
            $firstNode = array_shift($nodes);

            // if name is not a first node but another first node is their
            if ($firstNode != $name and isset($this->loggers[$firstNode])) {
                $logger->setParent($this->loggers[$firstNode]);
            } else {
                // if there is no father, set root logger as father
                $logger->setParent($this->root);
            }

            // if there are more nodes than one
            if (count($nodes) > 0) {
                // find parent node
                foreach ($nodes as $node) {
                    $parentNode = "$firstNode.$node";
                    if (isset($this->loggers[$parentNode]) and $parentNode != $name) {
                        $logger->setParent($this->loggers[$parentNode]);
                    }
                    $firstNode .= ".$node";
                }
            }

            $this->loggers[$name] = $logger;
        }

        return $this->loggers[$name];
    }

}
