<?php

namespace IZON\Logs\Deduplication\Tests;

use DateTimeImmutable;
use IZON\Logs\Deduplication\DeduplicationManagerInterface;
use IZON\Logs\Deduplication\SQLiteDeduplicationManager;
use Monolog\Logger;
use PHPUnit\Framework\TestCase;
use Psr\Log\LogLevel;
use Throwable;

class SQLiteDeduplicationManagerTest extends TestCase
{
    protected string $dbPath = __DIR__ . '/logs/log-deduplication.sqlite';

    public function testInitializeDeduplicationManager(): void
    {
        unlink($this->dbPath);
        try {
            $deduplicationManager = new SQLiteDeduplicationManager(
                $this->dbPath,
                10
            );
        } catch (Throwable $e) {
            $this->fail('Failed to initialize SQLiteDeduplicationManager');
        }
        $this->assertInstanceOf(DeduplicationManagerInterface::class, $deduplicationManager);

        try {
            $deduplicationManager = new SQLiteDeduplicationManager(
                $this->dbPath,
                10
            );
        } catch (Throwable $e) {
            $this->fail('Failed reconnect to existing db SQLiteDeduplicationManager');
        }
        $this->assertInstanceOf(DeduplicationManagerInterface::class, $deduplicationManager);
    }

    public function testExceptionSerialization(): void
    {
        $deduplicationManager = $this->createDeduplicationManager();

        $exception = new \Exception('test1');
        $records = [
            0 => $this->createLogRecord('channel1', 'test1', ['exception' => $exception])
        ];
        $deduplicated = $deduplicationManager->deduplicate($records);

        $this->assertEquals(1, count($deduplicated));
        $this->assertSameRecord($records[0], $deduplicated[0]);
        $this->assertEquals($exception->getMessage(), $deduplicated[0]['context']['exception']->getMessage());
        $this->assertEquals($exception->getLine(), $deduplicated[0]['context']['exception']->getLine());
    }

    /**
     * first call do not aggregate any records os manager only makes sure that last message of every type and channel is stored
     * to start aggregation window
     */
    public function testSingleCall(): void
    {
        $deduplicationManager = $this->createDeduplicationManager();

        $channel1 = 'channel1';
        $channel2 = 'channel2';
        $records = [
            0 => $this->createLogRecord($channel1, 'test1'),
            1 => $this->createLogRecord($channel1, 'test2'),
            2 => $this->createLogRecord($channel2, 'test2'),
            3 => $this->createLogRecord($channel1, 'test1'),
            4 => $this->createLogRecord($channel1, 'test3'),
            5 => $this->createLogRecord($channel2, 'test1'),
            6 => $this->createLogRecord($channel2, 'test2'),
        ];

        $deduplicated = $deduplicationManager->deduplicate($records);

        $this->assertEquals(7, count($deduplicated));
        foreach ($records as $index => $record) {
            $this->assertSameRecord($record, $deduplicated[$index]);
        }
    }

    /**
     * in first call pass all records as no record was stored before
     * in second call only pass that are different from first call
     */
    public function testStoreInSecondCall(): void
    {
        $deduplicationManager = $this->createDeduplicationManager();

        $channel1 = 'channel1';
        $channel2 = 'channel2';
        $records = [
            0 => $this->createLogRecord($channel1, 'test1'),
            1 => $this->createLogRecord($channel1, 'test2'),
            2 => $this->createLogRecord($channel2, 'test1'),
            3 => $this->createLogRecord($channel2, 'test2'),
        ];

        $deduplicated = $deduplicationManager->deduplicate($records);

        $this->assertEquals(4, count($deduplicated));
        foreach ($records as $index => $record) {
            $this->assertSameRecord($record, $deduplicated[$index]);
        }

        $secondCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1'),
            1 => $this->createLogRecord($channel1, 'test2'),
            3 => $this->createLogRecord($channel2, 'test1'),
            4 => $this->createLogRecord($channel2, 'test3'),
            5 => $this->createLogRecord($channel2, 'test2'),
        ];
        $secondDeduplicated = $deduplicationManager->deduplicate($secondCallRecords);
        $this->assertEquals(1, count($secondDeduplicated));
        $this->assertSameRecord($secondCallRecords[4], $secondDeduplicated[0]);
    }

    /**
     * in first call pass all records as no record was stored before
     * in second call do not pass any records as these are the same as in first call
     * in third call return all records from second call and all records from third call as aggregation window passed
     */
    public function testPopAllFromSecondCall(): void
    {
        $deduplicationManager = $this->createDeduplicationManager(1);

        $channel1 = 'channel1';
        $channel2 = 'channel2';

        $firstCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 1]),
            1 => $this->createLogRecord($channel2, 'test2', ['call' => 1]),
        ];
        $firstDeduplicated = $deduplicationManager->deduplicate($firstCallRecords);
        $this->assertEquals(2, count($firstDeduplicated));
        foreach ($firstCallRecords as $index => $record) {
            $this->assertSameRecord($record, $firstDeduplicated[$index]);
        }

        $secondCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 2]),
            1 => $this->createLogRecord($channel2, 'test2', ['call' => 2]),
        ];
        $secondDeduplicated = $deduplicationManager->deduplicate($secondCallRecords);
        $this->assertEquals(0, count($secondDeduplicated));

        sleep(2);

        $thirdCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 3]),
            1 => $this->createLogRecord($channel1, 'test2', ['call' => 3]),
            2 => $this->createLogRecord($channel2, 'test1', ['call' => 3]),
            3 => $this->createLogRecord($channel2, 'test3', ['call' => 3]),
            4 => $this->createLogRecord($channel2, 'test2', ['call' => 3]),
            5 => $this->createLogRecord($channel2, 'Červeňoučký kůň úpěl ďábelské ódy', ['call' => 3]),
        ];
        $thirdDeduplicated = $deduplicationManager->deduplicate($thirdCallRecords);
        $this->assertEquals(8, count($thirdDeduplicated));

        $this->assertSameRecord($secondCallRecords[0], $thirdDeduplicated[0]);
        $this->assertSameRecord($secondCallRecords[1], $thirdDeduplicated[1]);
        $this->assertSameRecord($thirdCallRecords[0], $thirdDeduplicated[2]);
        $this->assertSameRecord($thirdCallRecords[1], $thirdDeduplicated[3]);
        $this->assertSameRecord($thirdCallRecords[2], $thirdDeduplicated[4]);
        $this->assertSameRecord($thirdCallRecords[3], $thirdDeduplicated[5]);
        $this->assertSameRecord($thirdCallRecords[4], $thirdDeduplicated[6]);
        $this->assertSameRecord($thirdCallRecords[5], $thirdDeduplicated[7]);
    }

    /**
     * in first call pass all records as no record was stored before
     * in second call also pass all as aggregation window passed
     * in third call also pass all as aggregation window passed
     * in fourth only pass one record
     */
    public function testPopStoredInSecondCall(): void
    {
        $deduplicationManager = $this->createDeduplicationManager(1);

        $channel1 = 'channel1';
        $channel2 = 'channel2';

        $firstCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 1]),
            1 => $this->createLogRecord($channel2, 'test2', ['call' => 1]),
        ];
        $firstDeduplicated = $deduplicationManager->deduplicate($firstCallRecords);
        $this->assertEquals(2, count($firstDeduplicated));
        foreach ($firstCallRecords as $index => $record) {
            $this->assertSameRecord($record, $firstDeduplicated[$index]);
        }

        sleep(2);

        $secondCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 2]),
            1 => $this->createLogRecord($channel2, 'test2', ['call' => 2]),
        ];
        $secondDeduplicated = $deduplicationManager->deduplicate($secondCallRecords);
        $this->assertEquals(2, count($secondDeduplicated));
        foreach ($secondCallRecords as $index => $record) {
            $this->assertSameRecord($record, $secondDeduplicated[$index]);
        }

        sleep(2);

        $thirdCallRecords = [
            0 => $this->createLogRecord($channel2, 'test2', ['call' => 3]),
        ];
        $thirdDeduplicated = $deduplicationManager->deduplicate($thirdCallRecords);
        $this->assertEquals(1, count($thirdDeduplicated));
        foreach ($thirdCallRecords as $index => $record) {
            $this->assertSameRecord($record, $thirdDeduplicated[$index]);
        }

        $fourthCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 4]),
            1 => $this->createLogRecord($channel2, 'test2', ['call' => 4]),
        ];
        $fourthDeduplicated = $deduplicationManager->deduplicate($fourthCallRecords);
        $this->assertEquals(1, count($fourthDeduplicated));
        $this->assertSameRecord($fourthCallRecords[0], $fourthDeduplicated[0]);
    }

    public function testPopOldRecordsAndGc(): void
    {
        $deduplicationManager = $this->createDeduplicationManager(1);

        $channel1 = 'channel1';
        $channel2 = 'channel2';

        $firstCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 1]),
            1 => $this->createLogRecord($channel2, 'test2', ['call' => 1]),
        ];
        $firstDeduplicated = $deduplicationManager->deduplicate($firstCallRecords);

        $secondCallRecords = [
            0 => $this->createLogRecord($channel1, 'test1', ['call' => 2]),
            1 => $this->createLogRecord($channel2, 'test2', ['call' => 2]),
        ];
        $secondDeduplicated = $deduplicationManager->deduplicate($secondCallRecords);
        $this->assertEquals(0, count($secondDeduplicated));

        sleep(2);

        $oldRecords = $deduplicationManager->popOldRecords();
        $this->assertEquals(2, count($oldRecords));
        $this->assertSameRecord($secondCallRecords[0], $oldRecords[0]);
        $this->assertSameRecord($secondCallRecords[1], $oldRecords[1]);

        $deduplicationManager->gc();
    }

    // TODO: error handling

    protected function createDeduplicationManager(
        int $maxAggregationAge = 10
    ): DeduplicationManagerInterface {
        if (file_exists($this->dbPath)) {
            unlink($this->dbPath);
        }
        return new SQLiteDeduplicationManager(
            $this->dbPath,
            $maxAggregationAge
        );
    }

    /**
     * @param array{"message": string, "channel": string, "datetime": DateTimeImmutable, 'context': array<string, mixed>} $expected
     * @param array{"message": string, "channel": string, "datetime": DateTimeImmutable, 'context': array<string, mixed>} $actual
     * @return void
     */
    protected function assertSameRecord(array $expected, array $actual): void
    {
        $this->assertEquals($expected['message'], $actual['message'], 'message does not match');
        $this->assertEquals($expected['channel'], $actual['channel'], 'channel does not match');
        $this->assertEquals($expected['context'], $actual['context'], 'context does not match');
        $this->assertEquals($expected['datetime']->format('Uu'), $actual['datetime']->format('Uu'), 'datetime does not match');
    }

    /**
     * @param string $channel channel message was logged to
     * @param string $message logged message
     * @param array<string, mixed> $context
     * @param string $logLevel psr LogLevel
     * @return array{
     *      "message": string,
     *      "context": array<string, mixed>,
     *      "level": int,
     *      "level_name": string,
     *      "channel": string,
     *      "datetime": DateTimeImmutable,
     *      "extra": array<string, mixed>
     * }
     */
    protected function createLogRecord(
        string $channel,
        string $message,
        array $context = [],
        string $logLevel = LogLevel::ERROR
    ): array {
        $monologLevel = Logger::toMonologLevel($logLevel);
        return [
            'message' => $message,
            'context' => $context,
            'level' => $monologLevel,
            'level_name' => Logger::getLevelName($monologLevel),
            'channel' => $channel,
            'datetime' => new DateTimeImmutable(),
            'extra' => [],
        ];
    }

    protected function tearDown(): void
    {
        //        if (file_exists($this->dbPath)) {
        //            unlink($this->dbPath);
        //        };
    }
}
