<?php

namespace IZON\Logs\Deduplication;

use DateTimeImmutable;
use IZON\Utils\ArrayUtils;
use Monolog\Formatter\JsonFormatter;
use Monolog\Logger;
use PDO;
use Throwable;

class SQLiteDeduplicationManager implements DeduplicationManagerInterface
{
    /**
     * format to store DateTime of event in db
     */
    public const DATE_FORMAT = 'Y-m-d\TH:i:s.uP';

    /**
     * format to convert date to microseconds timestamp
     */
    public const TIMESTAMP_FORMAT = 'Uu';

    /**
     * @var string Path to the SQLite database file
     */
    protected string $dbPath;

    /**
     * @var string Table name of logs
     */
    protected string $logsTableName = 'logs';

    /**
     * @var int Max age in microseconds of records to be aggregated to one flush
     */
    protected int $maxAggregationAge;

    /**
     * @var int maximal number of old records to be dumped from aggregation database
     */
    protected int $maxOldRecordsCount;

    /**
     * @var bool true if connection to SQLite database is initialized
     */
    protected bool $initialized = false;

    /**
     * @var PDO|null PDO instance od SQLite connection
     */
    protected ?PDO $pdo = null;

    /**
     * @var JsonFormatter formatter to format records for serialization
     */
    protected JsonFormatter $formatter;

    /**
     * @var array<int, array{
     *     "message": string,
     *      "context": array<string, mixed>,
     *      "level": int,
     *      "level_name": string,
     *      "channel": string,
     *      "datetime": DateTimeImmutable,
     *      "extra": array<string, mixed>
     *   }> errors that deduplication manager generated itself
     */
    protected array $selfErrors = [];

    /**
     * @param string $dbPath Path to the SQLite database file to store logs
     * @param int $maxAggregationAge max age in seconds of records to be aggregated to one flush
     * @param int $maxOldRecordsCount max number of old records to be popped at once by popOldRecords method
     */
    public function __construct(
        string $dbPath,
        int $maxAggregationAge = 1 * 60 * 60,
        int $maxOldRecordsCount = 20
    ) {
        $this->dbPath = $dbPath;
        $this->maxAggregationAge = $maxAggregationAge * 1000000; // in miliseconds
        $this->maxOldRecordsCount = $maxOldRecordsCount;
        $this->formatter = new JsonFormatter();
        $this->formatter->includeStacktraces(true);
    }


    /**
     * @inheritDoc
     */
    public function deduplicate(array $records): array
    {
        $messageTypesHash = [];
        // hash according to channel and message
        foreach ($records as $record) {
            $channel = $record['channel'];
            if (!array_key_exists($channel, $messageTypesHash)) {
                $messageTypesHash[$channel] = [];
            }
            $message = $record['message'];
            if (!array_key_exists($message, $messageTypesHash[$channel])) {
                $messageTypesHash[$channel][$message] = [];
            }
            $messageTypesHash[$channel][$message][] = $record;
        }

        $deduplicated = [];
        foreach ($messageTypesHash as $channel => $channelSameMessages) {
            foreach ($channelSameMessages as $message => $sameMessages) {
                $deduplicatedRecords = $this->deduplicateRecordType(
                    $channel,
                    $message,
                    $sameMessages
                );
                $deduplicated = array_merge($deduplicated, $deduplicatedRecords);
            }
        }
        $deduplicated = array_merge($deduplicated, $this->selfErrors);

        // sort according to datetime of the record creation
        usort($deduplicated, function (array $a, array $b) {
            return $a['datetime'] <=> $b['datetime'];
        });

        return $deduplicated;
    }

    public function popOldRecords(): array
    {
        $connection = $this->getConnect();
        if ($connection === null) {
            return [];
        }

        try {
            $connection->beginTransaction();
        } catch (Throwable $e) {
            $this->logSelfErrors($e);
            return [];
        }

        try {
            $oldRecords = [];

            $statement = $connection->prepare("
                SELECT *
                FROM {$this->logsTableName}
                WHERE is_sent = 0
                    AND created_at < :created_at
                ORDER BY created_at ASC
            ");

            $now = new DateTimeImmutable();
            $dumpBefore = intval($now->format(self::TIMESTAMP_FORMAT)) - $this->maxAggregationAge;
            $statement->execute(
                [
                    'created_at' => $dumpBefore
                ]
            );

            $lastRecordTimestamp = null;
            $count = 0;
            while ($record = $statement->fetch(PDO::FETCH_ASSOC)) {
                if ($count >= $this->maxOldRecordsCount) {
                    break;
                }
                $oldRecords[] = $record;
                $lastRecordTimestamp = $record['created_at'];
                $count++;
            }

            // add records with same timestamp as last record
            if ($count >= $this->maxOldRecordsCount) {
                while (
                    $record = $statement->fetch(PDO::FETCH_ASSOC)
                    && $lastRecordTimestamp == $record['created_at']
                ) {
                    $oldRecords[] = $record;
                }
            }

            $oldRecords = $this->hydrateRecords($oldRecords);
            if (count($oldRecords) === 0) {
                $connection->commit();
                return [];
            }

            $statement = $connection->prepare("
                UPDATE {$this->logsTableName}
                SET is_sent = 1
                WHERE is_sent = 0
                    AND created_at <= :created_at
            ");
            $statement->execute([
                'created_at' => $lastRecordTimestamp
            ]);
        } catch (Throwable $e) {
            $connection->rollBack();
            $this->logSelfErrors($e);
            return [];
        }

        $connection->commit();
        return $oldRecords;
    }

    public function gc(): void
    {
        $connection = $this->getConnect();
        if ($connection === null) {
            return;
        }
        try {
            $statement = $connection->prepare("
                DELETE FROM {$this->logsTableName}
                WHERE is_sent = 1
                    AND created_at < :created_at
            ");

            $now = new DateTimeImmutable();
            $statement->execute(
                [
                    'created_at' => intval($now->format(self::TIMESTAMP_FORMAT)) - 2 * $this->maxAggregationAge,
                ]
            );
        } catch (Throwable $e) {
            $this->logSelfErrors($e);
        }
    }

    protected function getConnect(): ?PDO
    {
        if ($this->initialized) {
            return $this->pdo;
        }
        try {
            $pdo = new PDO('sqlite:' . $this->dbPath);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $pdo->setAttribute(PDO::ATTR_TIMEOUT, 5); // 5 second timeout for database lock
            $pdo->exec("
                CREATE TABLE IF NOT EXISTS {$this->logsTableName} (
                    channel TEXT,
                    message TEXT,
                    is_sent INTEGER DEFAULT 0, -- was already sent to the log
                    created_at INTEGER, -- timestamp in seconds message was created
                    json_data TEXT
                )
            ");
            $pdo->exec("
                CREATE INDEX IF NOT EXISTS {$this->logsTableName}_channel_message_index
                ON {$this->logsTableName} (channel, message, created_at)
            ");
            $pdo->exec("
                CREATE INDEX IF NOT EXISTS {$this->logsTableName}_created_at_index
                ON {$this->logsTableName} (is_sent, created_at)
            ");
            $this->pdo = $pdo;
        } catch (Throwable $e) {
            $this->logSelfErrors($e);
        }
        $this->initialized = true;
        return $this->pdo;
    }

    /**
     * deduplicates one record type (same message from same channel)
     * @param string $channel
     * @param string $message
     * @param array<int, array{
     *     "message": string,
     *      "context": array<string, mixed>,
     *      "level": int,
     *      "level_name": string,
     *      "channel": string,
     *      "datetime": DateTimeImmutable,
     *      "extra": array<string, mixed>
     *   }> $records
     * @return array<int, array{
     *      "message": string,
     *       "context": array<string, mixed>,
     *       "level": int,
     *       "level_name": string,
     *       "channel": string,
     *       "datetime": DateTimeImmutable,
     *       "extra": array<string, mixed>
     *   }>
     */
    protected function deduplicateRecordType(string $channel, string $message, array $records): array
    {
        $connection = $this->getConnect();
        if ($connection === null) {
            return $this->stampRecords($records, ['deduplicatedOverSQLite' => false]);
        }
        try {
            $connection->beginTransaction();
        } catch (Throwable $e) {
            $this->logSelfErrors($e);
            return $this->stampRecords($records, ['deduplicatedOverSQLite' => false]);
        }

        try {
            $storedRecords = $this->loadStoredRecords(
                $connection,
                $channel,
                $message
            );

            if (count($storedRecords) === 0) { // no records of this type stored
                $lastRecord = ArrayUtils::last($records);
                // store last record with to have info about last sent record
                $this->insertRecords(
                    $connection,
                    $channel,
                    $message,
                    [$lastRecord],
                    true
                );
                $connection->commit();
                return $this->stampRecords($records, ['deduplicatedOverSQLite' => true]);
            }

            $lastSentTimestamp = null;
            $notSentRows = [];
            foreach ($storedRecords as $row) {
                if ($row['is_sent'] == 1) {
                    $lastSentTimestamp = $row['created_at'];
                } else {
                    $notSentRows[] = $row;
                }
            }

            if ($lastSentTimestamp === null) { // no sent record stored in the database
                // handle first not set record as last sent
                $firsNotSend = ArrayUtils::first($notSentRows);
                $lastSentTimestamp = $firsNotSend['created_at'];
            }

            $now = new DateTimeImmutable();
            if ($lastSentTimestamp + $this->maxAggregationAge > intval($now->format(self::TIMESTAMP_FORMAT))) { // last sent record is not older than maxAggregationAge
                // store records to the database
                $this->insertRecords(
                    $connection,
                    $channel,
                    $message,
                    $records,
                    false
                );
                $connection->commit();
                return []; // do not return any record
            }

            // insert last records to the database
            $this->insertRecords(
                $connection,
                $channel,
                $message,
                [ArrayUtils::last($records)],
                true
            );

            $recordsToReturn = $this->hydrateRecords($notSentRows);
            $recordsToReturn = array_merge($recordsToReturn, $records);

            // mark all stored records from same channel and with same message as sent
            $statement = $connection->prepare("
                UPDATE {$this->logsTableName}
                SET is_sent = 1
                WHERE channel = :channel
                    AND message = :message
                    AND is_sent = 0
            ");
            $statement->execute([
                'channel' => $channel,
                'message' => $message
            ]);
        } catch (Throwable $e) {
            $this->logSelfErrors($e);
            $connection->rollBack();
            return $this->stampRecords($records, ['deduplicatedOverSQLite' => false]);
        }

        $connection->commit();

        return $this->stampRecords(
            $recordsToReturn,
            ['deduplicatedOverSQLite' => true]
        );
    }

    /**
     * @param PDO $connection
     * @param string $channel
     * @param string $message
     * @return array<int, array{"is_sent": string, "created_at": string, "json_data": string}>
     */
    protected function loadStoredRecords(PDO $connection, string $channel, string $message): array
    {
        $statement = $connection->prepare("
            SELECT is_sent, created_at, json_data
            FROM {$this->logsTableName}
            WHERE channel = :channel
                AND message = :message
            ORDER BY created_at ASC
        ");
        $statement->execute([
            'channel' => $channel,
            'message' => $message,
        ]);
        /** @var array<int, array{"is_sent": string, "created_at": string, "json_data": string}> $storedRecords */
        $storedRecords = $statement->fetchAll(PDO::FETCH_ASSOC);
        return $storedRecords;
    }

    /**
     * inserts records of provided type (same message from same channel) to the database
     * @param PDO $connection
     * @param string $channel
     * @param string $message
     * @param array<int, array{
     *     "message": string,
     *     "context": array<string, mixed>,
     *     "level": int,
     *     "level_name": string,
     *     "channel": string,
     *     "datetime": DateTimeImmutable,
     *     "extra": array<string, mixed>
     * }> $records
     * @param bool $isSent
     */
    protected function insertRecords(
        PDO $connection,
        string $channel,
        string $message,
        array $records,
        bool $isSent
    ): void {
        $i = 0;
        $valuesPlaceholders = [];
        $values = [
            "channel" => $channel,
            "message" => $message,
            "is_sent" => $isSent ? 1 : 0,
        ];
        foreach ($records as $record) {
            $recordDatetime = $record['datetime'];
            $record['datetime'] = $recordDatetime->format(self::DATE_FORMAT);
            $valuesPlaceholders[] = "(:channel, :message, :is_sent, :created_at_$i, :json_data_$i)";
            $values["created_at_$i"] = $recordDatetime->format(self::TIMESTAMP_FORMAT); // timestamp in microseconds
            $values["json_data_$i"] = $this->formatter->format($record);
            $i++;
        }
        $imploded = implode(', ', $valuesPlaceholders);

        $statement = $connection->prepare("
            INSERT INTO {$this->logsTableName} (channel, message, is_sent, created_at, json_data)
            VALUES $imploded
        ");
        $statement->execute($values);
    }

    /**
     * stamp records with extra context appended to $record['extra'] array
     * @param array<int, array{
     *     "message": string,
     *     "context": array<string, mixed>,
     *     "level": int,
     *     "level_name": string,
     *     "channel": string,
     *     "datetime": DateTimeImmutable,
     *     "extra"?: array<string, mixed>
     * }> $records
     * @param array<string, mixed> $extraContext
     * @return array<int, array{
     *      "message": string,
     *      "context": array<string, mixed>,
     *      "level": int,
     *      "level_name": string,
     *      "channel": string,
     *      "datetime": DateTimeImmutable,
     *      "extra": array<string, mixed>
     *  }> $records
     */
    protected function stampRecords(array $records, array $extraContext = []): array
    {
        foreach ($records as $recordIndex => $record) {
            if (!array_key_exists('extra', $record)) {
                $records[$recordIndex]['extra'] = [];
            }
            foreach ($extraContext as $key => $value) {
                $records[$recordIndex]['extra'][$key] = $value;
            }
            $records[$recordIndex]['extra']['deduplicated'] = true;
            $records[$recordIndex]['extra']['deduplicatedCount'] = count($records);
        }
        return $records;
    }

    protected function logSelfErrors(Throwable $ex): void
    {
        $this->selfErrors[] = [
            'datetime' => new \DateTimeImmutable(),
            'channel' => 'logs-deduplication',
            'level' => Logger::CRITICAL,
            'level_name' => Logger::getLevelName(Logger::CRITICAL),
            'message' => $ex->getMessage(),
            'context' => [
                'exception' => $ex,
            ],
            'extra' => [],
        ];
    }

    /**
     * @param array<int, array{"json_data": string}> $records
     * @return array<int, array{
     *     "message": string,
     *     "context": array<string, mixed>,
     *     "level": int,
     *     "level_name": string,
     *     "channel": string,
     *     "datetime": DateTimeImmutable,
     *     "extra": array<string, mixed>
     *  }> $records
     */
    protected function hydrateRecords(array $records): array
    {
        $recordsToReturn = array_map(function (array $record) {
            $recordData = json_decode($record['json_data'], true);
            $recordData['datetime'] = DateTimeImmutable::createFromFormat(self::DATE_FORMAT, $recordData['datetime']);
            return $recordData;
        }, $records);
        return $recordsToReturn;
    }
}
