<?php

namespace IZON\Logs\Deduplication;

use Monolog\Formatter\FormatterInterface;
use Monolog\Handler\BufferHandler;
use Monolog\Handler\FormattableHandlerInterface;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\ProcessableHandlerInterface;
use Monolog\Handler\ProcessableHandlerTrait;
use Monolog\Logger;

/**
 * handler that aggregate records between multiple runs of script
 * and sends them in one batch after specified time window passed
 */
class DeduplicationHandler extends BufferHandler implements ProcessableHandlerInterface, FormattableHandlerInterface
{
    use ProcessableHandlerTrait;

    protected DeduplicationManagerInterface $deduplicationManager;

    public function __construct(
        HandlerInterface $handler,
        DeduplicationManagerInterface $deduplicationManager,
        $bufferLimit = 0,
        $level = Logger::DEBUG,
        $bubble = true,
        $flushOnOverflow = false
    ) {
        parent::__construct($handler, $bufferLimit, $level, $bubble, $flushOnOverflow);
        $this->deduplicationManager = $deduplicationManager;
    }

    public function flush(): void
    {
        if ($this->bufferSize === 0) {
            return;
        }

        // deduplicate new records
        $deduplicated = $this->deduplicationManager->deduplicate($this->buffer);
        // default of null is valid as well as if no record matches duplicationLevel we just pass through
        if (!empty($deduplicated)) {
            // if we have old records out of deduplication window we pop them
            $old = $this->deduplicationManager->popOldRecords();
            // and merge old records with deduplicated records
            $deduplicated = array_merge($old, $deduplicated);
            // sort according to datetime of the record creation
            usort($deduplicated, function (array $a, array $b) {
                return $a['datetime'] <=> $b['datetime'];
            });
            $this->handler->handleBatch($deduplicated);
        }

        $this->clear();
        $this->deduplicationManager->gc();
    }

    /**
     * @inheritDoc
     */
    public function setFormatter(FormatterInterface $formatter): HandlerInterface
    {
        if ($this->handler instanceof FormattableHandlerInterface) {
            $this->handler->setFormatter($formatter);

            return $this;
        }

        throw new \UnexpectedValueException(
            'The nested handler of type ' . get_class($this->handler) . ' does not support formatters.'
        );
    }

    /**
     * @inheritDoc
     */
    public function getFormatter(): FormatterInterface
    {
        if ($this->handler instanceof FormattableHandlerInterface) {
            return $this->handler->getFormatter();
        }

        throw new \UnexpectedValueException(
            'The nested handler of type ' . get_class($this->handler) . ' does not support formatters.'
        );
    }
}
