<?php

namespace IZON\Gallery;

use \Exception;

class ImageResizer {

    public static $VERSION = '1.0';
    protected static $defaultThumbsdir = 'thumbs/';

    public function __construct() {
        throw new Exception('ImageResizer is static class, is imposibble to make an intance.', 0);
    }

    /**
     * zastresujici fce pro zmenu velikosti
     *
     * $params['width'] = jakou bude mit obrazek sirku
     * $params['height'] = jakou bude mit obrazek vysku
     * $params['thumbs'][] = array('dir'=> , 'width' =>, 'height' => );
     *
     * @param string $filename jmeno souboru
     * @param string $baseDir zdrojovy adresar
     * @param array $params parametry
     * @access public
     * @return bool
     */
    public static function run($filename, $baseDir, $params, $tweekFactor = 1.65) {
        try {
            $ret = false;
            if (file_exists($baseDir . $filename) && self::isPicture($baseDir . $filename)) {
                $original_size = getimagesize($baseDir . $filename);
                self::computeMemoryNeeds($original_size, $tweekFactor);

                // vytvareni nahledu
                if (is_array($params['thumbs'])) {
                    foreach ($params['thumbs'] as $key => $value) {
                        self::resizer($filename, self::getNewImageSize($original_size, array(0 => $value['width'], 1 => $value['height'])), $baseDir, $baseDir . ($value['dir'] != '' ? $value['dir'] : self::$defaultThumbsdir));
                    }
                }
                // zmena rozmeru puvodniho obrazku
                self::resizer($filename, self::getNewImageSize($original_size, array(0 => $params['width'], 1 => $params['height'])), $baseDir, $baseDir);
            } else {
                throw new ImageResizerException(__METHOD__ . ' > Source file doesn`t exists', 1);
            }
        } catch (Exception $e) {
            throw new ImageResizerException(__METHOD__ . ' > ' . $e->getMessage(), $e->getCode());
        }
    }

    public static function computeMemoryNeeds($size, $tweekFactor = 1.65) {
        $memoryneed = round(($size[0] * $size[1] * $size['bits'] * $size['channels'] / 8 + pow(2, 16)) * $tweekFactor);
        if (function_exists('memory_get_usage') && ((memory_get_usage() + $memoryneed) > ((integer) ini_get('memory_limit') * pow(1024, 2)))) {
            if (!ini_set('memory_limit', (integer) ini_get('memory_limit') + ceil(((memory_get_usage() + $memoryneed) - (integer) ini_get('memory_limit') * pow(1024, 2)) / pow(1024, 2)) . 'M')) {
                throw new ImageResizerException(__METHOD__ . ' > Don`t have enought memory for manipulation with entered picture. Required memory is at least ' . $memoryneed . '.', 3);
            }
        }
    }

    public static function isPicture($file) {
        $size = getimagesize($file);
        switch ($size[2]) {
            case IMAGETYPE_JPEG: return true;
                return true;
            case IMAGETYPE_PNG: return true;
                return true;
            case IMAGETYPE_GIF: return true;
                return true;
            default:
                throw new ImageResizerException(__METHOD__ . ' > Unsupported file type '. $size[2] .'.', 2);
        }
    }

    /**
     * Provede prepocet velikosti puvodniho obrazku na pozadovane rozmery. 
     *
     * @param array $original_size
     * @param array $required_size
     * @access public
     * @return array                        
     */
    public static function getNewImageSize($original_size, $required_size) {
        list($width_max_size, $height_max_size) = $required_size;
        list($width_original, $height_original) = $original_size;
        $height_max_size += 0;
        $width_max_size += 0;

        $new_height = 0;
        $new_width = 0;
        
        if ($width_max_size > 0 && $width_max_size < $width_original) {
          // pokud maximalni sirka nebude rovna nule a pokud bude max. sirka mensi nez sirka puvodniho obrazku
            $new_height = ($height_original * $width_max_size) / $width_original;
            $new_width = $width_max_size;

            return self::getNewImageSize(array(0 => floor($new_width), 1 => floor($new_height)), $required_size);
            
        } elseif ($height_max_size > 0 && $height_max_size < $height_original) {
          // pokud maximalni vyska nebude rovna nule a pokud bude max. vyska mensi nez vyska puvodniho obrazku
            $new_width = ($width_original * $height_max_size) / $height_original;
            $new_height = $height_max_size;

            return self::getNewImageSize(array(0 => floor($new_width), 1 => floor($new_height)), $required_size);
            
        } else {
          // nebudeme delat nic, tzn. nechame puvodni velikost obrazku
            return $original_size;
        }
    }
    
    /**
     * Provede prepocet velikosti puvodniho obrazku na pozadovane rozmery tak, aby se dalo prekryt pozadovanymy rozmery
     *
     * @param array $original_size
     * @param array $required_size
     * @access public
     * @return array                        
     */
    public static function getNewCoverImageSize($original_size, $required_size) {
        list($containerWidth, $containerHeight) = $required_size;
        list($imgWidth, $imgHeight)             = $original_size;
        
        // obrazek je alespon v jednom rozmeru mensi nez pozadovane, nema cenu resizovat
        if($imgWidth <= $containerWidth
           || $imgHeight <= $containerHeight) {
           return $original_size;
        }
        
        // http://stackoverflow.com/questions/10285134/whats-the-math-behind-csss-background-sizecover
        $imgRatio = ($imgHeight / $imgWidth);       // original img ratio
        $containerRatio = ($containerHeight / $containerWidth);     // container ratio
        
        if( $containerRatio > $imgRatio ) {
            $finalHeight = $containerHeight;
            $finalWidth = floor($containerHeight / $imgRatio);
        } else {
            $finalWidth = $containerWidth;
            $finalHeight = floor($containerWidth * $imgRatio); // oprava proti clanku
        }
        
        return [$finalWidth, $finalHeight];
    }

    /**
     * Wrapper pro resizer
     * 
     * @param type $sourceFile
     * @param type $sourceDir
     * @param type $targetDir
     * @param type $newSize
     * @param type $params - [targetFile]
     */
    public static function resizerParams($params = array()) {
        try {
            self::resizer($params['sourceFile'], $params['newSize'], $params['sourceDir'], $params['targetDir'], $params['targetFile'], $params['params']);
        } catch (Exception $exc) {
            throw $exc;
        }
    }

    /**
     * provede vlastni zmenu rozmeru obrazku
     *
     * @param string $filename      
     * @param array $new_size
     * @param string $sourceDir
     * @param string $targetDir
     * @access protected
     * @return bool            
     */
    public static function resizer($filename, $new_size, $sourceDir, $targetDir, $targetFileName = null, $params = array()) {
      if(!array_key_exists('quality', $params)) {
        $params['quality'] = 80;
      }
      $params['quality'] += 0; 
      if ($params['quality'] == 0) {
        $params['quality'] = 90;
      }
      if ($params['quality'] > 100) {
        $params['quality'] = 100;
      }


        if (empty($targetFileName))
            $targetFileName = $filename;
        list($new_width, $new_height) = $new_size;

        if (file_exists($sourceDir . $filename)) {
            $size = getimagesize($sourceDir . $filename);
            list($orig_width, $orig_height) = $size;
            $image_p = imagecreatetruecolor($new_width, $new_height);
        } else {
            throw new ImageResizerException(__METHOD__ . ' > Source file doesn`t exists', 1);
        }

        if( !file_exists($targetDir) ) { // test jestli existuje cilovy adresar
            mkdir($targetDir, 0777, true);
        }

        switch ($size[2]) {
            case IMAGETYPE_JPEG:
                $image = imagecreatefromjpeg($sourceDir . $filename);
                imagecopyresampled($image_p, $image, 0, 0, 0, 0, $new_width, $new_height, $orig_width, $orig_height);
                imagejpeg($image_p, $targetDir . $targetFileName, $params['quality']);
                break;
            case IMAGETYPE_PNG:
                imagealphablending($image_p, false);
                $image = imagecreatefrompng($sourceDir . $filename);
                imagecopyresampled($image_p, $image, 0, 0, 0, 0, $new_width, $new_height, $orig_width, $orig_height);
                imagesavealpha($image_p, true);
                $quality = round($params['quality'] / 10, 0);
                if ($quality > 9)
                    $quality = 9;
                if ($quality <= 0)
                    $quality = 0;
                imagepng($image_p, $targetDir . $targetFileName, $quality, PNG_ALL_FILTERS);
                break;
            case IMAGETYPE_GIF:
                $image = imagecreatefromgif($sourceDir . $filename);
                // Make the background transparent
                imagecolortransparent($image_p, imagecolorallocatealpha($image_p, 0, 0, 0, 127));
                imagecopyresampled($image_p, $image, 0, 0, 0, 0, $new_width, $new_height, $orig_width, $orig_height);
                imagegif($image_p, $targetDir . $targetFileName);
                break;
            default:
                throw new ImageResizerException(__METHOD__ . ' > Unsupported file type '. $size[2] .'.', 2);
        }
        imagedestroy($image);

    }

}

class ImageResizerException extends Exception {

    function __construct($message, $code = 0) {
        parent::__construct($message, $code);
    }

}
