<?php

namespace IZON\Forms\Fields;

use Exception;

/**
 * Combobox
 *
 * inspired by http://doc.nette.org/cs/2.3/forms#toc-addselect
 */
class SelectField extends BaseField {

    /**
     * ??? probably can be removed
     * String for empty select box
     * @var string
     */
    protected $unselectedText = NULL;

    /**
     * Options of select box
     * Associative array, returned value is key and value(item) is value for render
     * @var array
     */
    protected $options = [];

    /**
     * two dimensional array       ['label' => $label, 'optitons' => array same like option property]
     * @var array of option arrays
     */
    protected $optGroups = [];

    /**
     * @var callable
     */
    protected $keyLabelFunction = NULL;


    /**
     * SelectField constructor.
     * @param string $name
     * @param string $label
     * @param array $params
     */
    public function __construct(string $name, string $label = '', array $params = []) {
        $this->name         = $name;
        $this->label        = $label;
        $this->propertyName = $this->name;
        $this->errors       = array();

        return $this;
    }

    public function setValue($value) {
        parent::setValue($value);
        foreach($this->options as $key => $option) {
            if( $key == trim($value) ) {
                $this->options[$key]["selected"] = true;
            } else {
                $this->options[$key]["selected"] = false;
            }
        }
    }

    /**
     * @return mixed|null
     */
    public function parse() {
        if ($this->nullOnEmpty && empty(trim($this->value))) {
            return null;
        } else {
            return $this->options[trim($this->value)]["value"];
        }
    }

    /**
     * @param $value
     */
    public function serialize($value) {
        if( $value === null ) {
            return;
        }
        $keyLabelFunction = $this->keyLabelFunction;
        if( $keyLabelFunction === null ) {
            $key = $value;
        } else {
            $keyLabel = $keyLabelFunction($value, null);
            $key = $keyLabel["key"];
        }
        if(is_bool($key)) {
            $key = (int)$key;
        }
        if( array_key_exists($key, $this->options) ) {
            $this->options[$key]["selected"] = true;
            $this->value = $key;
        }
    }


    /**
     * @return string
     */
    function getUnselectedText(): ?string {
        return $this->unselectedText;
    }

    /**
     * @return bool
     */
    function hasUnselectedText(): bool {
        return !empty($this->unselectedText);
    }

    /**
     * @return array
     */
    function getOptions(): array {
        return $this->options;
    }

    /**
     * @param string $unselectedText
     * @return SelectField
     */
    function setUnselectedText(string $unselectedText) {
        $this->unselectedText = $unselectedText;
        return $this;
    }

    /**
     *
     * @param mixed[] $options array of options if $keyLabelFunction is not provided array of arrays ['key' =>, 'label' => ] is expected
     * @param callable $keyLabelFunction callback function to extract data from options from $options parameter must return ['key' =>, 'label' => ] array, returned value is value of element from $options, can be changes by providing 'value'
     * @return SelectField
     */
    function setOptions(iterable $options, $keyLabelFunction = null) {
        if( $keyLabelFunction !== null ) {
            if( !is_callable($keyLabelFunction) ) {
                throw new \Exception('$keyLabelFunction must be callable');
            }
            $this->keyLabelFunction = $keyLabelFunction;
        }

        $keyLabelFunction = $this->keyLabelFunction;
        if( $keyLabelFunction === null ) {
            $keyLabelFunction = function($option, $index) {
                if( !array_key_exists("key", $option) ) {
                    throw new Exception("\$option doesn't have element key");
                }
                if( !array_key_exists("label", $option) ) {
                    throw new Exception("\$option doesn't have element label");
                }
                $value = $option['key'];
                if( array_key_exists("value", $option) ) { // we pass value in
                    $value = $option["value"];
                }

                return ['key' => $option['key'], 'value' => $value, "label" => $option["label"]];
            };
        }

        $this->options = [];
        foreach($options as $index => $value) {
            $keyLabel = $keyLabelFunction($value, $index);
            if( !array_key_exists("key", $keyLabel) ) {
                throw new Exception("\$keyLableFunction doesn't return key for option");
            }
            if( !array_key_exists("label", $keyLabel) ) {
                throw new Exception("\$keyLableFunction doesn't return label for option");
            }
            if( array_key_exists("value", $keyLabel) ) { // we pass value in
                $value = $keyLabel["value"];
            }

            $this->options[$keyLabel["key"]] = [
                "value" => $value,
                "label" => $keyLabel["label"],
                "selected" => false,
            ];
        }

        return $this;
    }

    /**
     * @return bool
     */
    function hasOptions(): bool {
        return !empty($this->options);
    }

    /**
     * @param string $groupName
     * @param array $options
     * @return $this
     */
    public function addOptGroup(string $groupName, array $options) {
        $optGroup = ['label' => $groupName, 'options' => $options];
        $this->optGroups[] = $optGroup;
        return $this;
    }

    /**
     * @return array ['label', 'options' => [array of options]]
     */
    public function getOptGroups(): array{
        return $this->optGroups;
    }

    /**
     * @return bool
     */
    public function hasOptGroups(): bool {
        return !empty($this->optGroups);
    }

    /**
     *
     */
    public function __clone() {
        // naklonuje validacni pravidla
        $clonedRules = [];
        foreach($this->rules as $rule) {
            $clonedRule = clone $rule;
            $rule->setField($this);
            $clonedRules[] = $clonedRule;
        }
        $this->rules = $clonedRules;
    }

    protected function getSpecificSupportedProperties(): array {
        return [];
    }
}
