<?php

/**
 * funkce pro zjednoduseni vypisu formularovych poli
 */

namespace IZON\Forms;

use Exception;
use IZON\Forms\Fields\BaseField;
use IZON\Forms\Fields\CharField;
use IZON\Forms\Fields\CheckboxField;
use IZON\Forms\Fields\DateFiled;
use IZON\Forms\Fields\DateTimeFiled;
use IZON\Forms\Fields\EmailField;
use IZON\Forms\Fields\FloatField;
use IZON\Forms\Fields\HiddenField;
use IZON\Forms\Fields\IdField;
use IZON\Forms\Fields\IntegerField;
use IZON\Forms\Fields\PasswordField;
use IZON\Forms\Fields\PhoneField;
use IZON\Forms\Fields\RadioListField;
use IZON\Forms\Fields\SelectField;
use ReflectionClass;

if (!defined('LF')) {
    define('LF', PHP_EOL);
}

/**
 * vypise label prvek formulare
 * @param BaseField $field
 * @param array $params = array('for'=> identifikator, 'noColon'=>[true|false] - bez dvojtecky)
 * @return string
 * @deprecated is not necessary
 */
function getFieldLabel(BaseField $field, $params = [])
{
    $ret = '';

    $class = 'label--';
    $refClass = new ReflectionClass($field);
    switch ($refClass->getShortName()) {
        case 'CheckboxField':
            $class .= 'checkbox';
            break;
        case 'RadioListField':
            $class .= 'radio';
            break;
        case 'TextField':
            $class .= 'textarea';
            break;
        case 'CharField':
        case 'DateFiled':
        case 'DateTimeFiled':
            $class .= 'text';
            break;
        case 'EmailField':
            $class .= 'email';
            break;
        case 'FloatField':
        case 'IntegerField':
            $class .= 'number';
            break;
        case 'PasswordField':
            $class .= 'password';
            break;
        case 'SelectField':
            $class .= 'select';
            break;
    }

    if ($field->getUseLabel()) {
        $ret = '<label' .
            (!$params['noFor'] && $field->getId() != '' ? ' for="' . $field->getId() . '"' : '') .
            ' class="' . $class . (!is_null($field->getLabelClass()) ? ' ' . $field->getLabelClass() : '') . '"' .
            '>' . $field->getLabel() . ' </label>';
        if ($field->hasErrors()) {
            $ret .= ' <div class="message">' . implode('<br>', $field->getErrors()) . '</div>';
        }
    }

    return $ret;
}

/**
 * vypise formularove pole pro id
 * @param IdField $field
 * @return string
 * @deprecated use getField
 */
function getIdField(IdField $field)
{
    return '<input name="' . $field->getFormName() . '" value="' . $field->getValue() . '" type="hidden">';
}

/**
 * vypise formularove pole pro id
 * @param HiddenField $field
 * @return string
 * @deprecated use getField
 */
function getHiddenField(HiddenField $field)
{
    return '<input name="' . $field->getFormName() . '" value="' . clearString($field->getValue()) . '" type="hidden" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro kratky text
 * @param CharField $field
 * @return string
 * @deprecated use getField
 */
function getCharField(CharField $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" value="' . clearString($field->getValue()) . '" type="text" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro kratky text
 * @param PhoneField $field
 * @param array $params
 * @return string
 * @deprecated use getField
 */
function getPhoneField(PhoneField $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" value="' . clearString($field->getValue()) . '" type="tel" ' . $field->getAttributesAsString() . '>';
}

/**
 * TODO:
 * vypise formularove pole pro dlouhy text do textarea
 * @param CharField $field
 * @return string
 * @deprecated use getField
 */
function getTextField($field, $params = [])
{
    if (!$field->hasAttribute("rows")) { // neprepisovat atribut rows, pokud je uz nastaveny
        $field->addAttribute('rows', $field->getAttribute('row') + 0 > 0 ? $field->getAttribute('row') : 5);
    }
    return getFieldLabel($field, $params) .
        '<textarea name="' . $field->getFormName() . '" ' . $field->getAttributesAsString() . '>' . clearString($field->getValue()) . '</textarea>';
}

/**
 * vypisuje formularove pole pro checkbox
 * @param CheckboxField $field
 * @return string
 * @deprecated use getField
 */
function getCheckboxField(CheckboxField $field, $params = [])
{
    $id = $field->getFormName();

    if (!is_null($field->getId())) {
        $id = $field->getId();
    }
    $field->setId('checkbox--' . $id);

    //if(!isset($params['labelPosition'])) $params['labelPosition'] = 'right';

    $ret = '<input  name="' . $field->getFormName() . '" ' .
        ($field->getValue() == true ? 'checked="checked" ' : '') .
        $field->getAttributesAsString() . ' ' .
        ' type="checkbox" >';
    if ($params['labelPosition'] == 'left') {
        return getFieldLabel($field, $params) . $ret;
    } else {
        return $ret . getFieldLabel($field, $params);
    }
}

/**
 * vypise formularove pole pro date
 * @param DateFiled $field
 * @param array $params
 * @return string
 * @deprecated use getField
 */
function getDateFiled(DateFiled $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" value="' . $field->getValue() . '" type="text" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro date
 * @param DateTimeFiled $field
 * @param array $params
 * @return string
 * @deprecated use getField
 */
function getDateTimeFiled(DateTimeFiled $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" value="' . $field->getValue() . '" type="text" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro email
 * @param EmailField $field
 * @return string
 * @deprecated use getField
 */
function getEmailField(EmailField $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" value="' . $field->getValue() . '" type="email" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro desetinne cislo
 * @param FloatField $field
 * @return string
 * @deprecated use getField
 */
function getFloatField(FloatField $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" value="' . $field->getValue() . '" type="number" step="' . $field->getStep() . '" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro cele cislo
 * @param IntegerField $field
 * @return string
 * @deprecated use getField
 */
function getIntegerField(IntegerField $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" value="' . $field->getValue() . '" type="number" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro datum
 * @param PasswordField $field
 * @return string
 * @deprecated use getField
 */
function getPasswordField(PasswordField $field, $params = [])
{
    return getFieldLabel($field, $params) .
        '<input name="' . $field->getFormName() . '" type="password" ' . $field->getAttributesAsString() . '>';
}

/**
 * vypise formularove pole pro selectbox
 * @param SelectField $field
 * @return string
 * @deprecated use getField
 */
function getSelectField(SelectField $field, $params = [])
{
    $select = getFieldLabel($field) .
        '<select  name="' . $field->getFormName() . '" ' . $field->getAttributesAsString() . '>';

    foreach ($field->getOptions() as $key => $value) {
        if (is_array($value)) {
            $select .= '<optgroup label="' . $value["label"] . '">';

            foreach ($value["options"] as $optVal => $optLable) {
                $select .= '<option value="' . $optVal . '"' . ($optVal == $field->getValue() ? ' selected="selected"' : '') . '>'
                    . $optLable
                    . '</option>';
            }

            $select .= "</optgroup>";
        } else {
            $select .= '<option value="' . $key . '"' . ($key == $field->getValue() ? ' selected="selected"' : '') . '>'
                . $value
                . '</option>';
        }
    }

    $select .= '</select>';

    return $select;
}

/**
 * vypise formularove pole pro radio
 *
 * @param RadioListField $field
 * @return string
 * @deprecated use getField
 */
function getRadioListField(RadioListField $field, $params = [])
{
    $separator = $params['separator'] . '';
    $id = $field->getFormName();

    if (!is_null($field->getId())) {
        $id = $field->getId();
    }
    $field->setId('radio--' . $id);

    $ret = [];
    foreach ($field->getOptions() as $key => $value) {
        $ret[] = '<input ' .
            ($key == $field->getValue() ? 'checked="checked"' : '') .
            'value="' . $key . '" ' .
            'id="' . $field->getId() . '--' . $key . '" ' .
            'name="' . $field->getFormName() . '" ' .
            'type="radio" ' .
            $field->getAttributesAsString() .
            '>' .
            '<label for="' . $field->getId() . '--' . $key . '" class="label--radio">' . $value . ' </label>';
    }
    $params['noFor'] = true;
    return getFieldLabel($field, $params) . implode($separator, $ret);
}

/**
 * Vraci tag form + skryta pole<br />
 *
 * parametry odpovidaji atributum tagu form, napr. $params['method'] = 'post';
 *
 * $params['hidden'] - pole skrytych poli
 * @param array $params asoc pole s parametry
 * @return string
 * @author Vitezslav Jahn <jahn@izon.cz>
 * @deprecated write directly html tag <form enctype="multipart/form-data" method="post" action="..."> , hidden fields should be defined in Form object
 */
function getFormHeader(array $params = [])
{
    $params['method'] = isset($params['method']) ? $params['method'] : 'post';
    $params['enctype'] = isset($params['enctype']) ? $params['enctype'] : 'multipart/form-data';
    $params['role'] = isset($params['role']) ? $params['role'] : 'form';
    $params['hidden']['formAction'] = isset($params['hidden']['formAction']) ? $params['hidden']['formAction'] : 'post';
    $params['noLF'] = isset($params['noLF']) ? $params['noLF'] : false;
    //$params['hidden']['lang']       = isset($params['hidden']['lang']) ? $params['hidden']['lang'] : $_REQUEST['lang'];

    $attrs = [];
    foreach ($params as $key => $value) {
        if ($key == 'hidden') {
            continue;
        }
        $attrs[] = $key . '="' . $value . '"';
    }
    $ret = '<form ' . (implode(' ', $attrs)) . '>' . ($params['noLF'] ? '' : LF);
    if (is_array($params['hidden'])) {
        $ret .= '<div>' . ($params['noLF'] ? '' : LF);
        foreach ($params['hidden'] as $key => $value) {
            if ($value instanceof HiddenField) {
                $ret .= getHiddenField($value);
            } elseif ($value instanceof IdField) {
                $ret .= getIdField($value);
            } else {
                $ret .= '<input type="hidden" name="' . $key . '" value="' . $value . '">' . ($params['noLF'] ? '' : LF);
            }
        }
        $ret .= '</div>' . ($params['noLF'] ? '' : LF);
    }
    return $ret;
}

/**
 * ocisti/nahradi string o skarede znaky
 *
 * @param null|string $string
 */
function clearString($string)
{
    return htmlentities($string ?? '');
}

/**
 * Vrati html kod formularoveho prvku
 *
 * @param Form $form
 * @param string $fieldName nazev prvku
 * @param array $params pole s parametry<br>
 * $params['template'] = jaky se ma pouzit template - defaultne "default"<br>
 * $params['raw'] = true - pokud je treba vratit jen input bez labelu, defaultne false <br>
 * $params['noCaption'] = true - pokud je treba vratit input obaleny labelem ale bez textu, defaultne false <br>
 * $params['caption'] - parametry pro popisek pole - struktura obdobna jako u definice tempalte, jen v polich <br>
 * $params['wrapper'] - parametry pro wrapper pole - struktura obdobna jako u definice tempalte, jen v polich<br>
 * $params['field'] - parametry formularove pole - struktura obdobna jako u definice tempalte, jen v polich<br>
 * $params['error'] - parametry formularove pole - struktura obdobna jako u definice tempalte, jen v polich<br>
 * @return string
 */
function getField(Form $form, string $fieldName, array $params = [])
{
    $typeTemplatesDir = __BASE_DIR__  // @phpstan-ignore constant.notFound
        . '/app/web/form-flelds-templates';
    if (!array_key_exists('template', $params) || empty($params['template'])) {
        $params['template'] = 'default';
    }
    return FormFieldTemplate::getFiled($params['template'], $typeTemplatesDir, $form->getField($fieldName), $params);
}

/**
 * Return HTML code of form field
 *
 * @param Form $form
 * @param string $fieldName element name
 * @param array $params parameters field<br>
 * $params['template'] = name of tempalte - by default "default"<br>
 * $params['raw'] = true - if it is to be returned only input tag without label, by default false <br>
 * $params['noCaption'] = true - pokud je treba vratit input obaleny labelem ale bez textu, defaultne false <br>
 * $params['caption'] - parametry pro popisek pole - struktura obdobna jako u definice tempalte, jen v polich <br>
 * $params['wrapper'] - parametry pro wrapper pole - struktura obdobna jako u definice tempalte, jen v polich<br>
 * $params['field'] - parametry formularove pole - struktura obdobna jako u definice tempalte, jen v polich<br>
 * $params['error'] - parametry formularove pole - struktura obdobna jako u definice tempalte, jen v polich<br>
 * @return string
 * @deprecated use getField(..., ['template' => 'fadmin', ...]) instead
 */
function getFAdminField(Form $form, $fieldName, $params = [])
{
    $typeTemplatesDir = __BASE_DIR__  // @phpstan-ignore constant.notFound
        . '/app/web-admin/form-flelds-templates';
    if (!is_dir($typeTemplatesDir)) {
        // if not exists, use older possible path
        $typeTemplatesDir = __BASE_DIR__  // @phpstan-ignore constant.notFound
            . '/app/static/web-admin/form-flelds-templates';
    }
    if (!array_key_exists('template', $params) || empty($params['template'])) {
        $params['template'] = 'default';
    }
    return FormFieldTemplate::getFiled(
        $params['template'],
        $typeTemplatesDir,
        $form->getField($fieldName),
        $params
    );
}

/**
 *
 * $params[label] = nastaveni labelu <br>
 *  -> [after]   - label zobrazit az za inputem<br>
 *  -> [class]   - trida(y) pro label<br>
 *  -> [for]     - string pro vlastnost "for"<br>
 *  -> [noColon] - nepouzivej dvoutecku<br>
 *
 * @param Form $form
 * @param string $fieldName
 * @param array $params
 * @return string
 * @throws Exception
 * @category Experimental
 */
function _getField(Form $form, $fieldName, $params = [])
{
    $field = $form->getField($fieldName);
    if (array_key_exists('isHidden', $params) && $params['isHidden']) {
        $type = '\IZON\Forms\getHiddenField';
    } else {
        $refClass = new ReflectionClass($field);
        $type = '\IZON\Forms\get' . $refClass->getShortName();
    }
    //echo $type.'<br>';
    if (!function_exists($type)) {
        throw new Exception(_('Pro zadany typ pole (' . $type . ') neexistuje renderovaci fce.'));
    }
    $useLabel = $field->getUseLabel();  // @phpstan-ignore-line method.deprecated
    // label se ma pouzit a nema se vypisovat za prvkem
    if (!$params['label']['after'] && $useLabel) {
        $ret = _getFieldLabel($field, $params['label']);
    }
    $ret .= call_user_func_array($type, [$field->setUseLabel(false), $params]); // @phpstan-ignore-line method.deprecated
    if ($params['label']['after'] && $useLabel) {
        $ret .= _getFieldLabel($field, $params['label']);
    }
    return $ret;
}

/**
 * vypise label prvek formulare
 *
 * @param BaseField $field
 * @param array $params parametry pro vykresleni<br>[class]- css trida(y) pro label<br>[for] - string pro vlastnost "for"
 *
 * @return string
 * @category Experimental
 */
function _getFieldLabel(BaseField $field, $params = [])
{
    $ret = '<label' .
        ($params['for'] != '' ? ' for="' . $params['for'] . '"' : '') .
        ($params['class'] ? ' class="' . $params['class'] . '"' : '') .
        '>' . $field->getLabel() . ' </label>';
    if ($field->hasErrors()) {
        $ret .= ' <div class="message">' . implode('<br>', $field->getErrors()) . '</div>';
    }

    return $ret;
}
