<?php

namespace IZON\Forms;

use Exception;
use IZON\Logs\Logger;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use function IZON\Object\createGetterMethodName;
use function IZON\Object\createSetterMethodName;

/**
 * Skupina nekolika formuaru.
 * Je mozne vyuzit v nekolikastrankovych wizardech nebo vlozeni nekolika stejnych formularu najednou napriklad nekolik ckanku
 */
class FormSet implements FormInterface
{
    /**
     * @var array<string> prefixy u nadrazenych form setu
     */
    protected array $prependedPrefix = [];

    /**
     * prefix slouzi k zanoreni do pole predavaneho v set values a ke spravnemu vypisu
     * formName pro jednotliva pole ve formulari
     * @var array<int, int|string> prefix pro vsechny pole formulare
     */
    protected array $prefix;

    /**
     * @var array<string, array{form: Form|FormSet, objectClass: class-string, propertyName: string}> formulare, ktere se mohou v setu pobjevit jen 1x
     */
    protected array $forms;

    /**
     * @var array<string, array{formDefinition: Form|FormSet, objectClass: class-string, params: array<string, mixed>, forms: array<int|string, Form|FormSet>, propertyName: string}> formulare, ktere se mohou v setu pobjevit objevit vicekrat
     */
    protected array $dynamicForms;

    /**
     * validator pro cely formular
     * muze provadet validace zavisle na vice polich
     * @var null|callable
     */
    protected $validator = null;

    protected LoggerInterface $log;

    /**
     * @var array<string>
     */
    protected array $errors = [];

    public function __construct(array $prefix = [])
    {
        $this->forms = [];
        $this->dynamicForms = [];

        $this->prefix = $prefix;

        $this->log = Logger::getLogger(__CLASS__);
    }

    /**
     * add form present in FormSet only once
     * @param Form|FormSet $form
     * @param null|class-string $objectClass
     * @param string[] $prefix pripadny prefix pro dany formular, pokud ma furmular jiz nastaveny prefix tak ho prepise
     * @param array{propertyName?: string} $params parametry pro praci s formularem
     */
    public function addForm(FormInterface $form, ?string $objectClass, array $prefix = [], array $params = []): void
    {
        $prefixString = $this->serializePrefix($prefix);

        if ($objectClass !== null && !class_exists($objectClass)) {
            throw new Exception("Class $objectClass doesn't exist.");
        }

        $propertyName = null;
        if (isset($params["propertyName"])) {
            $propertyName = $params["propertyName"];
        } elseif (count($prefix) > 0) {
            $propertyName = $prefix[0];
        }

        if (array_key_exists($prefixString, $this->forms)) {
            throw new Exception('Form with prefix ' . $prefixString . ' is already used.', 100);
        } elseif (array_key_exists($prefixString, $this->dynamicForms)) {
            throw new Exception('DynamicForm with prefix ' . $prefixString . ' is already used.', 200);
        } else {
            $form->setPrefix(array_merge($this->getPrefix(), $prefix));
            if ($form instanceof FormSet) {
                $form->setPrependedPrefix($this->prefix);
            }
            $this->forms[$prefixString] = [
                "form" => $form,
                "objectClass" => $objectClass,
                "propertyName" => $propertyName,
            ];
        }
    }

    /**
     * serializuje prefix do stringu, pod kterym se uklada do vnitrnih promennych setu
     * @param array<int, int|string> $prefix
     * @return string
     */
    protected function serializePrefix(array $prefix): string
    {
        $prefixString = '';
        if ($prefix != null) {
            $prefixString = "[" . implode(",", $prefix) . "]";
        }
        return $prefixString;
    }

    /**
     * @return string[] reprezentuje preffix
     */
    public function getPrefix(): array
    {
        return $this->prefix;
    }

    /**
     * @param string[] $prefix
     * @return void
     */
    public function setPrefix(array $prefix): void
    {
        $this->prefix = $prefix;
        $this->prependPrefixToSubForms(array_merge($prefix), true);
    }

    /**
     * @param string[] $prependedPrefix
     * @return void
     */
    public function setPrependedPrefix(array $prependedPrefix): void
    {
        $this->prefix = array_merge($prependedPrefix, array_splice($this->prefix, count($this->prependedPrefix)));
        $this->prependedPrefix = $prependedPrefix;
        foreach ($this->forms as $prefixString => $formInfo) {
            $form = $formInfo["form"];
            if ($form instanceof FormSet) {
                $form->setPrependedPrefix($this->prefix);
            }
        }
        // spracuje dinamicke formulare
        foreach ($this->dynamicForms as $prefixString => $dinForms) {
            $formDef = $dinForms["formDefinition"];
            if ($formDef instanceof FormSet) {
                $formDef->setPrependedPrefix($this->prefix);
            }
        }
    }

    /**
     * vratit formular podle prefixu, pokud neni prefix specializovan tak vraci formular, ktery byl pridan bez prefixu
     *
     * @param string[] $prefix
     * @return Form|FormSet form stored under $prefix
     */
    public function getForm(array $prefix = []): FormInterface
    {
        $prefixString = $this->serializePrefix($prefix);

        if (!array_key_exists($prefixString, $this->forms)) {
            throw new Exception('Formular s prefixem  ' . $prefixString . ' neexistuje.');
        } else {
            return $this->forms[$prefixString]["form"];
        }
    }

    /**
     * @return array<string, Form|FormSet> returns all frorms that are used only once
     */
    public function getForms(): array
    {
        $forms = [];
        foreach ($this->forms as $prefix => $formInfo) {
            $forms[$prefix] = $formInfo["form"];
        }
        return $forms;
    }

    /**
     * @param Form|FormSet $form form template to be cloned
     * @deprecated use addDynamicFormDefinition instead
     */
    public function addDinamicFormDefinition(FormInterface $form, $objectClass, array $prefix, $params = []): void
    {
        $this->addDynamicFormDefinition($form, $objectClass, $prefix, $params);
    }

    /**
     * prida formular, ktery je mozne vlozit vicekrat napriklad nekolik kategorii/autoru k jednomu clanku
     *
     * $formFactory callback funkce je ve tvaru function($preffix, $formIndex)
     * $prefix, ktery se ma pro formular pouzivat
     * $formIndex index, kterym se odlisuji instance formularu
     *
     * @param Form|FormSet $form form template to be cloned
     * @param null|class-string $objectClass objekty jake tridy se maji doplnit pokud pocet prvku v property je mene nez formularu
     * @param string[] $prefix prefix, ktery se ma pouzit pred vsemi formulari
     * @param array{propertyName?: string, collectionFlushFunction?: callable} $params parametry minCount - minimalni pocet opakovani, maxCount - maximalni pocet opakovani, collectionFlushFunction = function(array $formsArray, $collection, $objectClass, $accessType) - funkce slouzisi ke zpracovani kolekce, da ze vyuzit, aby se obsahy formularu vlozily do spravnych objektu kolekce a doplnili se do dostatecneho poctu, musi bracet upravenou kolekci
     */
    public function addDynamicFormDefinition(
        FormInterface $form,
        ?string $objectClass,
        array $prefix,
        array $params = []
    ): void {
        $prefixString = $this->serializePrefix($prefix);

        if ($objectClass !== null && !class_exists($objectClass)) {
            throw new Exception("Class $objectClass doesn't exist.");
        }

        $propertyName = null;
        if (isset($params["propertyName"])) {
            $propertyName = $params["propertyName"];
        } elseif (count($prefix) > 0) {
            $propertyName = $prefix[0];
        }

        if (array_key_exists($prefixString, $this->forms)) {
            throw new Exception('Form with prefix ' . $prefixString . ' is already used.', 100);
        } elseif (array_key_exists($prefixString, $this->dynamicForms)) {
            throw new Exception('DynamicForm with prefix ' . $prefixString . ' is already used.', 200);
        } else {
            //            var_dump(get_class($form));
            //            var_dump($prefix);
            $form->setPrefix(array_merge($this->getPrefix(), $prefix));
            if ($form instanceof FormSet) {
                $form->setPrependedPrefix($this->prefix);
            }
            $this->dynamicForms[$prefixString] = [
                "formDefinition" => $form, // contains template of form
                "objectClass" => $objectClass,
                "params" => $params, //
                "forms" => [], // actual forms with data
                "propertyName" => $propertyName,
            ];
        }
    }

    /**
     * @deprecated use addDynamicFormDefinition instead
     */
    public function getDinamicFormDefinition(array $prefix)
    {
        return $this->getDynamicFormDefinition($prefix);
    }

    /**
     * @param array<string> $prefix vrati formular odpovidejici definici dynamickych formularu
     * @return Form|FormSet definition of dynamic from
     */
    public function getDynamicFormDefinition(array $prefix): FormInterface
    {
        $prefixString = $this->serializePrefix($prefix);

        if (array_key_exists($prefixString, $this->dynamicForms)) {
            return $this->dynamicForms[$prefixString]["formDefinition"];
        } else {
            throw new Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * @deprecated use getDynamicFormDefinitions instead
     */
    public function getDinamicFormDefinitions(): array
    {
        return $this->getDynamicFormDefinitions();
    }

    /**
     * @return array<string, Form|FormSet> vrati pole definici dynamickych formularu
     */
    public function getDynamicFormDefinitions(): array
    {
        $dynamicFormsDefinitions = [];
        foreach ($this->dynamicForms as $prefix => $dinFormInfo) {
            $dynamicFormsDefinitions[$prefix] = $dinFormInfo["formDefinition"];
        }
        return $dynamicFormsDefinitions;
    }

    /**
     * @deprecated use createDynamicForm instead
     */
    public function createDinamicForm(array $prefix, $index): FormInterface
    {
        return $this->createDynamicForm($prefix, $index);
    }

    /**
     * vytvori novy dinamicly formular ulozeny pod prefiex ktery bude mit prefix - array_mege($prefix, [$index])
     * da se pouzit pro pridavani novych formularu do stranky
     * @param array $prefix
     * @param string|int $index
     * @return Form|FormSet
     */
    public function createDynamicForm(array $prefix, $index): FormInterface
    {
        $prefixString = $this->serializePrefix($prefix);

        if (array_key_exists($prefixString, $this->dynamicForms)) {
            $formDef = $this->dynamicForms[$prefixString]["formDefinition"];
            $form = clone $formDef;
            $form->setPrefix(array_merge($prefix, [$index]));
            return $form;
        } else {
            throw new Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * @deprecated use getDynamicFormsCount instead
     */
    public function getDinamicFormsCount(array $prefix): int
    {
        return $this->getDynamicFormsCount($prefix);
    }

    /**
     * @param string[] $prefix number of dynamic forms with given prefix
     */
    public function getDynamicFormsCount(array $prefix): int
    {
        $prefixString = $this->serializePrefix($prefix);

        if (array_key_exists($prefixString, $this->dynamicForms)) {
            return count($this->dynamicForms[$prefixString]["forms"]);
        } else {
            throw new Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * @deprecated use getDynamicForms instead
     */
    public function getDinamicForms(array $prefix): array
    {
        return $this->getDynamicForms($prefix);
    }

    /**
     * @param string[] $prefix
     * @return array<int|string, Form|FormSet> array of all dynamic forms with given prefix
     */
    public function getDynamicForms(array $prefix): array
    {
        $prefixString = $this->serializePrefix($prefix);

        if (array_key_exists($prefixString, $this->dynamicForms)) {
            return $this->dynamicForms[$prefixString]["forms"];
        } else {
            throw new Exception('DynamicForm with prefix ' . $prefixString . ' doesn\'t exist.', 100);
        }
    }

    /**
     * sets validator for whole SormSet
     * @param callable $validator is a method with one parame FormSet $formSet, it sets found errors tirectly to $form
     */
    public function setValidator($validator): void
    {
        if (!is_callable($validator)) { // @phpstan-ignore function.alreadyNarrowedType
            throw new Exception("Validator must be callable");
        }
        $this->validator = $validator;
    }

    /**
     * {@inheritDoc}
     */
    public function validate(): void
    {
        // validuje normalni formulare
        foreach ($this->forms as $form) {
            $form["form"]->validate();
        }
        // validuje dinamicke formulare
        foreach ($this->dynamicForms as $dynamicForm) {
            foreach ($dynamicForm["forms"] as $form) {
                $form->validate();
            }
        }
        // vola validator na cely FormSet
        if (is_callable($this->validator)) {
            $validator = $this->validator;
            $validator($this);
        }
    }

    /**
     * @return bool true if dynamic forms or normal forms have errors or if there are errors set directly to FormSet
     */
    public function hasErrors(): bool
    {
        $hasErrors = false;
        foreach ($this->forms as $form) {
            $hasErrors |= $form["form"]->hasErrors();
        }
        // validuje dinamicke formulare
        foreach ($this->dynamicForms as $dynamicForm) {
            foreach ($dynamicForm["forms"] as $form) {
                $hasErrors |= $form->hasErrors();
            }
        }
        return $hasErrors || !empty($this->errors);
    }

    /**
     * @return bool true if there are errors set directly to FormSet, not bound to any forms of dynamic forms
     */
    public function hasFormSetErrors(): bool
    {
        return !empty($this->errors);
    }

    /**
     * prida chybu navazanou primo na formulari ne navazane na formularove pole
     * @param string $errorMessage
     */
    public function addFormSetError(string $errorMessage): void
    {
        $this->errors[] = $errorMessage;
    }

    /**
     * vrati chyby navazane primo na formulari ne navazane na formularove pole
     * @return string[]
     */
    public function getFormSetErrors(): array
    {
        return $this->errors;
    }

    /**
     * {@inheritdoc}
     */
    public function setValues($array): void
    {
        // handle normal forms
        foreach ($this->forms as $prefixString => $formInfo) {
            $form = $formInfo["form"];
            $same = $this->getSamePrefix($form->getPrefix());
            $prefix = array_merge($this->prefix, array_slice($form->getPrefix(), count($same)));
            $form->setPrefix($prefix);
            $form->setValues($array);
        }
        // handle dynamic forms
        foreach ($this->dynamicForms as $prefixString => $dynamicForm) {
            $formDef = $dynamicForm["formDefinition"];
            $same = $this->getSamePrefix($formDef->getPrefix());
            $prefix = array_merge($this->prefix, array_slice($formDef->getPrefix(), count($same)));
            $prefixedArray = $array;
            foreach ($prefix as $pref) { // presun se na prefix
                if (isset($prefixedArray[$pref])) {
                    $prefixedArray = $prefixedArray[$pref];
                } else {
                    $prefixedArray = [];
                }
            }
            foreach ($prefixedArray as $index => $valuesArray) {
                $form = clone $formDef;
                $form->setPrefix(array_merge($prefix, [(string)$index]));
                $form->setValues($array);
                $this->dynamicForms[$prefixString]["forms"][$index] = $form;
            }
        }
    }

    protected function getSamePrefix(array $prefix): array
    {
        $same = [];
        $max = min(count($prefix), count($this->getPrefix()));
        for ($i = 0; $i < $max; $i++) {
            if ($prefix[$i] === $this->getPrefix()[$i]) {
                $same[] = $prefix[$i];
            }
        }
        return $same;
    }

    /**
     * {@inheritdoc}
     */
    public function getValues(): array
    {
        $return = [];

        // handle static forms
        foreach ($this->forms as $prefixString => $formInfo) {
            /** @var FormInterface $form */
            $form = $formInfo["form"];
            $formValues = $form->getValues();
            $return = array_merge($return, $formValues);
        }

        // handle dynamic forms
        foreach ($this->dynamicForms as $prefixString => $dynamicForm) {
            $forms = $dynamicForm["forms"];
            $formsValues = [];
            foreach ($forms as $form) {
                $formValues = $form->getValues();
                $formsValues = $this->mergeArrayRecursive($formsValues, $formValues);
            }
            $return = $this->mergeArrayRecursive($return, $formsValues);
        }

        // add prefix
        foreach ($this->prefix as $prefixPart) {
            $return = [$prefixPart => $return];
        }
        return $return;
    }

    protected function mergeArrayRecursive(array &$array1, array $array2)
    {
        foreach ($array2 as $key => $value) {
            if (array_key_exists($key, $array1)) { // is in first array
                if (is_array($array1[$key])
                    && is_array($array2[$key])) { // source and destination values are arrays
                    $array1[$key] = $this->mergeArrayRecursive($array1[$key], $array2[$key]);
                } else {
                    $array1[$key] = $array2[$key];
                }
            } else {
                $array1[$key] = $array2[$key];
            }
        }
        return $array1;
    }

    /**
     * ze zadaneho objektu a podpobjektu nastavi hodnoty jednotlivych formularu
     * @param object $object
     * @param FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS|FormInterface::EDIT_FLUSH_ACCESS_PROPERTIES $accessType how to access properties of object EDIT_FLUSH_ACCESS_GETTERS_SETTERS - through getters/setters , EDIT_FLUSH_ACCESS_PROPERTIES - directly to propertis
     */
    public function edit($object, $accessType = FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS): void
    {
        if (is_null($object)) { // @phpstan-ignore function.impossibleType
            throw new Exception("Object to edit data from must not be null");
        }

        $reflectionClass = new ReflectionClass($object);

        // projit normalni formulare
        foreach ($this->forms as $formInfo) {
            /** @var Form $form */
            $form = $formInfo['form'];
            $propertyName = $formInfo["propertyName"];
            if ($propertyName == null) { // mapuje se primo na  $object
                $form->edit($object, $accessType);
                continue;
            }

            if ($accessType == FormInterface::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if ($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $subObject = $reflectionProperty->getValue($object);
                    if ($subObject != null) {
                        $form->edit($subObject);
                    }
                } elseif (
                    $reflectionClass->hasMethod("__isset") // ma metodu isset
                    && $object->__isset($propertyName) // ma property nadefinovane pomoci __isset, __get a __set metod
                ) {
                    $subObject = $object->$propertyName;
                    if ($subObject != null) {
                        $form->edit($subObject);
                    }
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to edit data from");
                }
            } elseif ($accessType == FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = createGetterMethodName($propertyName);
                if ($reflectionClass->hasMethod($getterName)) {
                    $getterMethod = $reflectionClass->getMethod($getterName);

                    $subObject = $getterMethod->invoke($object);
                    if ($subObject != null) {
                        $form->edit($subObject);
                    }
                } else {
                    $this->log->info("Doesn't have getter: " . $getterName . " to edit data from");
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }

        // projit dinamicke formulare, musi se mapovat na array nebo na kolekci
        foreach ($this->dynamicForms as $dynamicFormIndex => $dynamicFormInfo) {
            $formDefinition = $dynamicFormInfo['formDefinition'];
            $propertyName = $dynamicFormInfo["propertyName"];

            if ($accessType == Form::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if ($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $collection = $reflectionProperty->getValue($object);
                    foreach ($collection as $index => $element) { // pro vsechny emementy vytvor pole
                        $clonedForm = clone $formDefinition;
                        $clonedForm->setPrefix(array_merge($formDefinition->getPrefix(), [$index]));
                        $clonedForm->edit($element, $accessType);
                        $this->dynamicForms[$dynamicFormIndex]["forms"][$index] = $clonedForm;
                    }
                } elseif (
                    $reflectionClass->hasMethod("__isset") // ma metodu isset
                    && $object->__isset($propertyName) // ma property nadefinovane pomoci __isset, __get a __set metod
                ) {
                    $collection = $object->$propertyName;
                    foreach ($collection as $index => $element) { // pro vsechny emementy vytvor pole
                        $clonedForm = clone $formDefinition;
                        $clonedForm->setPrefix(array_merge($formDefinition->getPrefix(), [$index]));
                        $clonedForm->edit($element, $accessType);
                        $this->dynamicForms[$dynamicFormIndex]["forms"][$index] = $clonedForm;
                    }
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to edit data from");
                }
            } elseif ($accessType == Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = createGetterMethodName($propertyName);
                if ($reflectionClass->hasMethod($getterName)) {
                    $getterMethod = $reflectionClass->getMethod($getterName);

                    $collection = $getterMethod->invoke($object);
                    foreach ($collection as $index => $element) { // pro vsechny emementy vytvor pole
                        $clonedForm = clone $formDefinition;
                        $clonedForm->setPrefix(array_merge($formDefinition->getPrefix(), [$index]));

                        if ($clonedForm instanceof FormSet) {
                            $clonedForm->prependPrefixToSubForms(
                                array_merge($formDefinition->getPrefix(), [$index]),
                                true
                            );
                        }
                        $this->dynamicForms[$dynamicFormIndex]["forms"][$index] = $clonedForm;

                        $clonedForm->edit($element, $accessType);
                    }
                } else {
                    $this->log->info("Doesn't have getter : " . $getterName . " to fedit data from");
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }
    }

    protected function prependPrefixToSubForms(array $prefix, $removeSamePrefix = false)
    {
        foreach ($this->forms as $prefixString => $formInfo) {
            $form = $formInfo["form"];
            if ($removeSamePrefix) {
                $same = $this->getSamePrefix($form->getPrefix());
                $form->setPrefix(array_merge($prefix, array_slice($form->getPrefix(), count($same))));
            } else {
                $form->setPrefix(array_merge($prefix, $form->getPrefix()));
            }
        }
        // spracuje dinamicke formulare
        foreach ($this->dynamicForms as $prefixString => $dinForms) {
            $formDef = $dinForms["formDefinition"];

            if ($removeSamePrefix) {
                $same = $this->getSamePrefix($formDef->getPrefix());
                $formDef->setPrefix(array_merge($prefix, array_slice($formDef->getPrefix(), count($same))));
            } else {
                $formDef->setPrefix(array_merge($prefix, $formDef->getPrefix()));
            }
        }
    }

    /**
     * z hodnot ve formularich nastavi hodnoty properit v objektu a podobjektech
     * @param object $object
     * @param FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS|FormInterface::EDIT_FLUSH_ACCESS_PROPERTIES $accessType how to access properties of object EDIT_FLUSH_ACCESS_GETTERS_SETTERS - through getters/setters , EDIT_FLUSH_ACCESS_PROPERTIES - directly to propertis
     */
    public function flush($object, $accessType = FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS): void
    {
        if (is_null($object)) { // @phpstan-ignore function.impossibleType
            throw new Exception("Object to flush data to must not be null");
        }

        $reflectionClass = new ReflectionClass($object);

        // handle normal forms
        foreach ($this->forms as $formInfo) {
            /** @var Form $form */
            $form = $formInfo['form'];
            $propertyName = $formInfo["propertyName"];
            if ($propertyName == null) { // mapuje se primo na  $object
                $form->flush($object, $accessType);
                continue;
            }

            if ($accessType == Form::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if ($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $subObject = $reflectionProperty->getValue($object);
                    if ($subObject == null) { // pokud je objekt null
                        $objectClass = $formInfo["objectClass"]; // objekty jake tridy se ma doplnovat
                        $subObject = new $objectClass();
                    }
                    $form->flush($subObject);
                    $reflectionProperty->setValue($object, $subObject);
                } elseif (
                    $reflectionClass->hasMethod("__isset") // ma metodu isset
                    && $object->__isset($propertyName) // ma property nadefinovane pomoci __isset, __get a __set metod
                ) {
                    $subObject = $object->$propertyName;
                    if ($subObject == null) {
                        $objectClass = $formInfo["objectClass"]; // objekty jake tridy se ma doplnovat
                        $subObject = new $objectClass();
                    }
                    $form->flush($subObject);
                    $object->$propertyName = $subObject;
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to flush data to");
                }
            } elseif ($accessType == FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = createGetterMethodName($propertyName);
                $setterName = createSetterMethodName($propertyName);
                if (
                    $reflectionClass->hasMethod($getterName) // mame getter
                    && $reflectionClass->hasMethod($setterName) // mame setter
                ) {
                    $getterMethod = $reflectionClass->getMethod($getterName);
                    $setterMethod = $reflectionClass->getMethod($setterName);

                    $subObject = $getterMethod->invoke($object);
                    if ($subObject == null) { // pokud je objekt null
                        $objectClass = $formInfo["objectClass"]; // objekty jake tridy se ma doplnovat
                        $subObject = new $objectClass();
                    }

                    $form->flush($subObject);
                    // nastavi kolekci zpet do objektu pomoci setteru
                    $setterMethod->invoke($object, $subObject);
                } else {
                    $this->log->info(
                        "Doesn't have getter: " . $getterName . " and setter: " . $setterName . " to flush data to"
                    );
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }

        // handle dynamic collections, must be mapped to array or collection
        foreach ($this->dynamicForms as $dynamicFormIndex => $dynamicFormInfo) {
            $forms = $dynamicFormInfo['forms'];
            $propertyName = $dynamicFormInfo["propertyName"];

            if ($accessType == Form::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
                if ($reflectionClass->hasProperty($propertyName)) {
                    $reflectionProperty = $reflectionClass->getProperty($propertyName);
                    $reflectionProperty->setAccessible(true);

                    $collection = $reflectionProperty->getValue($object);
                    $collection = $this->handleFlushCollection($forms, $collection, $dynamicFormInfo, $accessType);
                    $reflectionProperty->setValue($object, $collection);
                } elseif ($reflectionClass->hasMethod("__isset") // ma metodu isset
                    && $object->__isset(
                        $propertyName
                    )) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $collection = $object->$propertyName;
                    $collection = $this->handleFlushCollection($forms, $collection, $dynamicFormInfo, $accessType);
                    $object->$propertyName = $collection;
                } else {
                    $this->log->info("Doesn't have property: " . $propertyName . " to flush data to");
                }
            } elseif ($accessType == Form::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
                $getterName = createGetterMethodName($propertyName);
                $setterName = createSetterMethodName($propertyName);
                if (
                    $reflectionClass->hasMethod($getterName) // we have getter method
                    && $reflectionClass->hasMethod($setterName) // we have setter method
                ) {
                    $getterMethod = $reflectionClass->getMethod($getterName);
                    $setterMethod = $reflectionClass->getMethod($setterName);
                    $collection = $getterMethod->invoke($object); // ziskat kolekci objektu

                    $collection = $this->handleFlushCollection($forms, $collection, $dynamicFormInfo, $accessType);

                    // nastavi kolekci zpet do objektu pomoci setteru
                    $setterMethod->invoke($object, $collection);
                }
            } else {
                throw new Exception("Unsupported access type " . $accessType);
            }
        }
    }

    /**
     *
     * @param iterable<Form|FormSet> $forms
     * @param iterable $collection
     * @param array $dinaFormInfo
     * @param FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS|FormInterface::EDIT_FLUSH_ACCESS_PROPERTIES $accessType
     * @return iterable<Form|FormSet>
     * @throws Exception
     */
    protected function handleFlushCollection(iterable $forms, $collection, $dinaFormInfo, $accessType): iterable
    {
        $objectClass = $dinaFormInfo["objectClass"]; // objekty jake tridy se ma doplnovat

        if (isset($dinaFormInfo["params"]["collectionFlushFunction"])) { // ma nastavenou funkci pro zpracovani kolekce
            $callback = $dinaFormInfo["params"]["collectionFlushFunction"];
            if (!is_callable($callback)) { // collectionFlushCallback neni funkce
                throw new Exception("collectionFlushFunction must be callable.");
            }

            // zpracuje kolekci podle vstupu z formularu
            $collection = $callback($forms, $collection, $objectClass, $accessType);

            if (!is_array($collection) && !is_iterable($collection)) {
                throw new Exception("collectionFlushFunction must return collection/array.");
            }
        } else {
            $formsArrayKeys = array_keys($forms);
            if (is_array($collection)) {
                $collectionArrayKeys = array_keys($collection);
            } else {
                $collectionArrayKeys = [];
                foreach ($collection as $key => $value) {
                    $collectionArrayKeys[] = $key;
                }
            }

            // smaž z kolekce co není ve formuláři
            $keysDelete = array_diff($collectionArrayKeys, $formsArrayKeys);
            foreach ($keysDelete as $key) {
                unset($collection[$key]);
            }
            // projdi formulare a vyber z kolekce
            foreach ($formsArrayKeys as $index) {
                // jaky furmular se ma flushovat
                $form = $forms[$index];
                if (!isset($collection[$index])) { // v kolekci neexituje objekt
                    $element = new $objectClass();
                } else {
                    $element = $collection[$index];
                    unset($collection[$index]);
                }
                // flushnout data do objektu
                $form->flush($element, $accessType);
                $collection[$index] = $element;
            }
        }

        return $collection;
    }

    public function __clone()
    {
        // spracuje normalni formulare
        foreach ($this->forms as $prefixString => $formInfo) {
            $formDef = $formInfo["form"];
            $this->forms[$prefixString]["form"] = clone $formDef;
        }
        // spracuje dinamicke formulare
        foreach ($this->dynamicForms as $prefixString => $dinForms) {
            $formDef = $dinForms["formDefinition"];
            $this->dynamicForms[$prefixString]["formDefinition"] = clone $formDef;
        }
    }
}
