<?php

namespace IZON\Forms;

use Exception;
use IZON\DB\EntityManagerInterface;
use IZON\Forms\Fields\BaseField;
use IZON\Forms\Fields\CharField;
use IZON\Forms\Fields\CheckboxField;
use IZON\Forms\Fields\CheckboxListField;
use IZON\Forms\Fields\DateFiled;
use IZON\Forms\Fields\DateTimeFiled;
use IZON\Forms\Fields\EmailField;
use IZON\Forms\Fields\EntityHiddenField;
use IZON\Forms\Fields\FileField;
use IZON\Forms\Fields\FloatField;
use IZON\Forms\Fields\FloatRangeField;
use IZON\Forms\Fields\HiddenField;
use IZON\Forms\Fields\IdField;
use IZON\Forms\Fields\ImageField;
use IZON\Forms\Fields\IntegerField;
use IZON\Forms\Fields\IntegerRangeField;
use IZON\Forms\Fields\MultipleEmailField;
use IZON\Forms\Fields\PasswordField;
use IZON\Forms\Fields\PhoneField;
use IZON\Forms\Fields\RadioListField;
use IZON\Forms\Fields\SelectField;
use IZON\Forms\Fields\TextField;
use IZON\Logs\Logger;
use Psr\Log\LoggerInterface;
use ReflectionClass;
use function IZON\Object\createGetterMethodName;
use function IZON\Object\createIsMethodName;
use function IZON\Object\createSetterMethodName;

/**
 * Trida formulare
 */
class Form implements FormInterface
{
    /**
     * jednotliva pole formulare
     * asociativni pole, klice jsou name poli, klice jsou jednotliva pole
     * @var array<string, BaseField> BaseField fole formulare
     */
    protected array $fields = [];

    /**
     * chyby primo vazane na formular
     * @var string[]
     */
    protected array $errors = [];

    /**
     * validator pro cely formular
     * muze provadet validace zavisle na vice polich
     * @var null|callable
     */
    protected $validator = null;

    /**
     * prefix slouzi k zanoreni do pole predavaneho v set values a ke spravnemu vypisu
     * formName pro jednotliva pole ve formulari
     * @var array<int, int|string> prefix pro vsechny pole formulare
     */
    protected array $prefix;

    protected LoggerInterface $log;


    public function __construct(array $prefix = [])
    {
        $this->prefix = $prefix;

        $this->log = Logger::getLogger(__CLASS__);
    }

    /**
     * @return string[] reprezentuje preffix
     */
    public function getPrefix(): array
    {
        return $this->prefix;
    }

    public function setPrefix(array $prefix): void
    {
        $this->prefix = $prefix;
        foreach ($this->fields as $field) {
            $field->setPrefix($prefix);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function setValues(array $array): void
    {
        $prefixedArray = $array;
        foreach ($this->prefix as $pref) {
            if (!isset($prefixedArray[$pref])) {
                $prefixedArray = [];
            } else {
                $prefixedArray = $prefixedArray[$pref];
            }
        }
        foreach ($this->fields as $name => $field) {
            if (isset($prefixedArray[$name])) {
                $field->setValue($prefixedArray[$name]);
            } else {
                $field->setValue(null);
            }
        }
    }

    /**
     * {@inheritdoc}
     */
    public function getValues(): array
    {
        $return = [];
        foreach ($this->fields as $name => $field) {
            $return[$name] = $field->getValue();
        }
        // add prefix
        foreach (array_reverse($this->prefix) as $prefixPart) {
            $return = [$prefixPart => $return];
        }
        return $return;
    }

    /**
     * {@inheritDoc}
     */
    public function validate(): void
    {
        // validuje jednotliva pole
        foreach ($this->fields as $name => $field) {
            $field->validate();
        }
        // vola validator celeho formulare
        if (is_callable($this->validator)) {
            $validator = $this->validator;
            $validator($this);
        }
    }

    public function hasFormErrors(): bool
    {
        return !empty($this->errors);
    }

    /**
     * prida chybu navazanou primo na formulari ne navazane na formularove pole
     * @param string $errorMessage
     */
    public function addFormError(string $errorMessage): void
    {
        $this->errors[] = $errorMessage;
    }

    /**
     * vrati chyby navazane primo na formulari ne navazane na formularove pole
     * @return string[]
     */
    public function getFormErrors(): array
    {
        return $this->errors;
    }

    public function clearFormErrors(): void
    {
        $this->errors = [];
    }

    /**
     * vrati formularove pole
     * @param string $name
     * @return BaseField
     */
    public function getField(string $name): BaseField
    {
        if (!isset($this->fields[$name])) {
            throw new Exception("Pole $name není definováno.");
        } else {
            return $this->fields[$name];
        }
    }

    /**
     * Chceck if field exist in form
     * @param string $name name of field
     * @return bool True - field exist False - field is not exist
     */
    public function haveField(string $name): bool
    {
        if (isset($this->fields[$name])) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return IdField
     */
    public function addIdField(string $fieldName, string $fieldLabel = ''): IdField
    {
        $field = new IdField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     *
     * prida do formulare jakekoli pole
     * @template Field of BaseField
     * @param Field $filed
     * @return Field
     * @throws Exception
     * @deprecated use addField
     */
    public function addFiled(BaseField $filed): BaseField
    {
        return $this->addField($filed);
    }

    /**
     *
     * prida do formulare jakekoli pole
     * @template Field of BaseField
     * @param Field $filed
     * @return Field
     * @throws Exception
     */
    public function addField(BaseField $filed): BaseField
    {
        if (!array_key_exists($filed->getName(), $this->fields)) {
            $this->fields[$filed->getName()] = $filed;
            $filed->setPrefix($this->prefix);
            return $this->fields[$filed->getName()];
        } else {
            throw new Exception('Field ' . $filed->getName() . ' already exists in form');
        }
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakywebhom jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return HiddenField
     */
    public function addHiddenField(string $fieldName, string $fieldLabel = ''): HiddenField
    {
        $field = new HiddenField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return CharField
     */
    public function addCharField(string $fieldName, string $fieldLabel = ''): CharField
    {
        $field = new CharField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Prida do formulare dlouhe textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return TextField
     */
    public function addTextField(string $fieldName, string $fieldLabel = ''): TextField
    {
        $field = new TextField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return CheckboxField
     */
    public function addCheckboxField(string $fieldName, string $fieldLabel = ''): CheckboxField
    {
        $field = new CheckboxField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Prida do formulare textove pole, wrapper pro addField
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return PasswordField
     */
    public function addPasswordField(string $fieldName, string $fieldLabel = ''): PasswordField
    {
        $field = new PasswordField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return DateFiled Description
     */
    public function addDateFiled(string $fieldName, string $fieldLabel = ''): DateFiled
    {
        $field = new DateFiled($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     *
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return DateTimeFiled Description
     */
    public function addDateTimeFiled(string $fieldName, string $fieldLabel = ''): DateTimeFiled
    {
        $field = new DateTimeFiled($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Adds file
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return FileField
     * @deprecated use different implementation
     */
    public function addFileField(string $fieldName, string $fieldLabel = ''): FileField
    {
        $field = new FileField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Adds file
     * @param string $fieldName
     * @param string $fieldLabel
     * @return ImageField
     * @deprecated use different implementation
     */
    public function addImageField(string $fieldName, string $fieldLabel = ''): ImageField
    {
        $field = new ImageField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Adds intefer field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return FloatField
     */
    public function addFloatField(string $fieldName, string $fieldLabel = ''): FloatField
    {
        $field = new FloatField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Adds imteger field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return IntegerField
     */
    public function addIntegerField(string $fieldName, string $fieldLabel = ''): IntegerField
    {
        $field = new IntegerField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Adds file
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return EmailField
     */
    public function addEmailField(string $fieldName, string $fieldLabel = ''): EmailField
    {
        $field = new EmailField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Adds multiple email field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return MultipleEmailField
     */
    public function addMultipleEmailField(string $fieldName, string $fieldLabel = ''): MultipleEmailField
    {
        $field = new MultipleEmailField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * Adds field
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel label pro prvek
     * @return PhoneField
     */
    public function addPhoneField(string $fieldName, string $fieldLabel = ''): PhoneField
    {
        $field = new PhoneField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     *  pridava select pole
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return SelectField
     */
    public function addSelectField(string $fieldName, string $fieldLabel = ''): SelectField
    {
        $field = new SelectField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     *  pridava radio pole
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return RadioListField
     */
    public function addRadioListField(string $fieldName, string $fieldLabel = ''): RadioListField
    {
        $field = new RadioListField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     *  adds array of checkoxes with multiple possible selected options
     * @param string $fieldName pod jakym jmenem ma byt hodnota ve formulari ulozena
     * @param string $fieldLabel prod jakym nazvem se ma
     * @return CheckboxListField
     */
    public function addCheckboxListField(string $fieldName, string $fieldLabel = ''): CheckboxListField
    {
        $field = new CheckboxListField($fieldName, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * pole integeru, ktere ma rozsah odnot od nejmensi do nejvetsi
     * @param string $fieldName
     * @param int $minValue
     * @param int $maxValue
     * @param string $fieldLabel
     * @return IntegerRangeField
     */
    public function addIntegerRangeField(string $fieldName, $minValue, $maxValue, string $fieldLabel = ''): IntegerRangeField
    {
        $field = new IntegerRangeField($fieldName, $minValue, $maxValue, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    /**
     * pole floatu, ktere ma rozsah odnot od nejmensi do nejvetsi
     * @param string $fieldName
     * @param integer $minValue
     * @param integer $maxValue
     * @param string $fieldLabel
     * @return FloatRangeField
     */
    public function addFloatRangeField(string $fieldName, $minValue, $maxValue, string $fieldLabel = ''): FloatRangeField
    {
        $field = new FloatRangeField($fieldName, $minValue, $maxValue, $fieldLabel);
        $this->addField($field);
        return $field;
    }

    public function addEntityHiddenField(string $fieldName, EntityManagerInterface $entityManager, string $entityClass): EntityHiddenField
    {
        $field = new EntityHiddenField($fieldName, $entityManager, $entityClass);
        $this->addField($field);
        return $field;
    }

    /**
     * ze zadaneho objektu nastavi hodnoty jednotlivych formularovych poli
     * @param mixed $object z jakeho objektu se maji ziskat data
     * @param FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS|FormInterface::EDIT_FLUSH_ACCESS_PROPERTIES $accessType Jakym zpusobem se ma pristupovat k property v objektu. Jestli primo nebo pres gettery a settery, default je dvuli kompatibilite pres gettery a settery
     */
    public function edit($object, $accessType = self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS): void
    {
        if (is_null($object)) {
            throw new Exception("Objekt ze ktereho se data kopiruji nesmi byt null");
        }

        $reflectionClass = new ReflectionClass($object);

        if ($accessType == self::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
            foreach ($this->fields as $key => $field) {
                if ($reflectionClass->hasProperty($field->getPropertyName())) {
                    $reflectionProperty = $reflectionClass->getProperty($field->getPropertyName());
                    $reflectionProperty->setAccessible(true);
                    $field->serialize($reflectionProperty->getValue($object));
                } elseif (
                    method_exists($object, "__isset")
                    && $object->__isset($field->getPropertyName()) // ma property nadefinovane pomoci __isset, __get a __set metod
                ) {
                    $propertyName = $field->getPropertyName();
                    $field->serialize($object->$propertyName);
                } else {
                    $this->log->info("Doesn't have property: " . $field->getPropertyName() . " to edit");
                }
            }
        } elseif ($accessType == self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
            foreach ($this->fields as $key => $field) {
                $propertyName = $field->getPropertyName();
                $getterName = createGetterMethodName($propertyName);
                $isName = createIsMethodName($propertyName);

                if ($reflectionClass->hasMethod($getterName)) { // has getter method get data through it
                    $getterMethod = $reflectionClass->getMethod($getterName);
                    $value = $getterMethod->invoke($object);
                    $field->serialize($value);
                } elseif ($reflectionClass->hasMethod($isName)) {
                    $getterMethod = $reflectionClass->getMethod($isName);
                    $value = $getterMethod->invoke($object);
                    $field->serialize($value);
                }
                if (
                    $reflectionClass->hasMethod("__call") // has magic method __call to be able to call getter method implemented by __call method
                    && $reflectionClass->hasMethod("__isset") // has method __isset to determine if setter for property should be provided by __call method
                    && $object->__isset($propertyName)  // has property so __call should provide getter
                ) {
                    $value = $object->__call($getterName, []);
                    $field->serialize($value);
                } else {
                    $this->log->info("Doesn't have getter: " . $getterName);
                }
            }
        } else {
            throw new Exception("Unsupported access type " . $accessType);
        }
    }

    /**
     *
     * @param object $object
     * @param FormInterface::EDIT_FLUSH_ACCESS_GETTERS_SETTERS|FormInterface::EDIT_FLUSH_ACCESS_PROPERTIES $accessType Jakym zpusobem se ma pristupovat k property v objektu. Jestli primo nebo pres gettery a settery, default je dvuli kompatibilite pres gettery a settery
     * @throws Exception
     */
    public function flush($object, $accessType = self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS): void
    {
        if (is_null($object)) { // @phpstan-ignore function.impossibleType
            throw new Exception("Object to flush data to must not be null");
        }

        $reflectionClass = new ReflectionClass($object);

        if ($accessType == self::EDIT_FLUSH_ACCESS_PROPERTIES) { // pristupovat pres property
            foreach ($this->fields as $key => $field) {
                if ($field->isOneWay()) { // to not map data back to object if field is one way
                    continue;
                }
                if ($reflectionClass->hasProperty($field->getPropertyName())) {
                    $reflectionProperty = $reflectionClass->getProperty($field->getPropertyName());
                    $reflectionProperty->setAccessible(true);
                    $reflectionProperty->setValue($object, $field->parse());
                } elseif (
                    $reflectionClass->hasMethod("__isset") // ma metodu isset
                    && $object->__isset($field->getPropertyName())
                ) { // ma property nadefinovane pomoci __isset, __get a __set metod
                    $propertyName = $field->getPropertyName();
                    $object->$propertyName = $field->parse();
                } else {
                    $this->log->info("Doesn't have property: " . $field->getPropertyName() . " to flush data to");
                }
            }
        } elseif ($accessType == self::EDIT_FLUSH_ACCESS_GETTERS_SETTERS) { // pristupovat pres gettery a settery
            foreach ($this->fields as $key => $field) {
                if ($field->isOneWay()) { // to not map data back to object if field is one way
                    continue;
                }
                $propertyName = $field->getPropertyName();
                $setterName = createSetterMethodName($propertyName);

                if ($reflectionClass->hasMethod($setterName)) { // ma setter metodu nastavit hodnotu pres ni
                    $setterMethod = $reflectionClass->getMethod($setterName);
                    $setterMethod->invoke($object, $field->parse());
                } elseif (
                    $reflectionClass->hasMethod("__call") // has magic method __call to be able to call getter method implemented by __call method
                    && $reflectionClass->hasMethod("__isset") // has method __isset to determine if setter for property should be provided by __call method
                    && $object->__isset($field->getPropertyName())  // has property so __call should provide getter
                ) {
                    $object->__call($setterName, [$field->parse()]);
                } else {
                    $this->log->info("Doesn't have setter: " . $setterName);
                }
            }
        } else {
            throw new Exception("Unsupported access type " . $accessType);
        }
    }

    /**
     * sets validator for whole form
     * @param callable $validator is a method with one parame Form $form, it sets found errors tirectly to $form
     */
    public function setValidator($validator): void
    {
        if (!is_callable($validator)) { // @phpstan-ignore function.alreadyNarrowedType
            throw new Exception("Validator must be callable");
        }
        $this->validator = $validator;
    }

    /**
     * @return array<int, array{formName: string, errors: string[]}>
     * @deprecated use getFieldErrors instead
     */
    public function getErrorsArray(): array
    {
        $errors = [];
        foreach ($this->getFields() as $fild) {
            if ($fild->hasErrors()) {
                $errorsArray = $fild->getErrors();
                $errors[] = ["formName" => $fild->getFormName(), "errors" => $errorsArray];
            }
        }
        return $errors;
    }

    /**
     * Varci pole zaregistrovanych formularovych poli
     * @return BaseField[]
     * @deprecated use getFields
     */
    public function getFileds(): array
    {
        return $this->getFields();
    }

    /**
     * returns all registered fields in form
     * @return BaseField[]
     */
    public function getFields(): array
    {
        return $this->fields;
    }

    /**
     * zjisti jestli ma formular nebo nejake formularove pole chybu
     * @return bool
     */
    public function hasErrors(): bool
    {
        $hasErrors = false;
        /** @var BaseField $field */
        foreach ($this->fields as $name => $field) {
            $hasErrors = $hasErrors || $field->hasErrors();
        }
        return $hasErrors || !empty($this->errors);
    }

    /**
     * returns array of errors for all form fields
     * @return array<int, array{"formName": string, "errors": string[]}>
     */
    public function getFieldErrors(): array
    {
        $errors = [];
        foreach ($this->getFields() as $field) {
            if ($field->hasErrors()) {
                $errorsArray = $field->getErrors();
                $errors[] = ["formName" => $field->getFormName(), "errors" => $errorsArray];
            }
        }
        return $errors;
    }

    /**
     * Smazne pole z formulare
     *
     * @param string $key - index, pod kterym bylo pole pridano
     */
    public function removeField($key): void
    {
        unset($this->fields[$key]);
    }

    public function __clone()
    {
        // naklonuje formularova pole
        $clonedFields = [];
        foreach ($this->fields as $index => $field) {
            $clonedField = clone $field;
            $clonedFields[$index] = $clonedField;
        }
        $this->fields = $clonedFields;
    }
}
