<?php

namespace IZON\Forms\Fields;

use Exception;

/**
 * Combobox
 *
 * inspired by http://doc.nette.org/cs/2.3/forms#toc-addselect
 */
class SelectField extends BaseField
{
    public const OPTION_TYPE = 'option';

    public const OPTGROUP_TYPE = 'optgroup';

    /**
     * text pro nevyplneno
     * @var null|string
     */
    protected ?string $unselectedText = null;

    /**
     * Options that represent possible values to be selected
     * Associative array, returned value is key and value(item) is value for render
     * @var array<string, array>
     */
    protected array $options = [];

    /**
     * structure that represents how to display options
     * element structure ['type' => option|optgroup, ...]
     * if option ['type' => ... , 'key' => 'key option is stored under in $options']
     * if optgroup ['type' => ... , 'label' => 'label of optgroup', 'optionKeys' => [ keys option is stored under in $options ]]
     * @var array<array>
     */
    protected array $optionsStructure = [];

    /**
     * @var null|callable
     */
    protected $keyLabelFunction = null;

    /**
     * @var callable
     */
    protected $defaultKeyLabelFunction;

    public function __construct(string $name, string $label = '')
    {
        $this->name = $name;
        $this->label = $label;
        $this->propertyName = $this->name;
        $this->errors = [];

        $this->defaultKeyLabelFunction = function ($option, $index) {
            if (!array_key_exists("key", $option)) {
                throw new Exception("\$option doesn't have element key");
            }
            if (!array_key_exists("label", $option)) {
                throw new Exception("\$option doesn't have element label");
            }
            $value = $option['key'];
            if (array_key_exists("value", $option)) { // we pass value in
                $value = $option["value"];
            }
            $escape = true;
            if (array_key_exists("escape", $option)) { // we pass value in
                $escape = $option["escape"];
            }

            return [
                'key' => $option['key'],
                'value' => $value,
                "label" => $option["label"],
                'escape' => $escape
            ];
        };
    }

    public function setValue($value)
    {
        parent::setValue($value);

        foreach ($this->options as $key => $option) {
            if ($value === null) {
                $this->options[$key]["selected"] = false;
                continue;
            }
            if ($key == trim($value)) {
                $this->options[$key]["selected"] = true;
            } else {
                $this->options[$key]["selected"] = false;
            }
        }
        return $this;
    }

    /**
     * @return mixed|null
     */
    public function parse()
    {
        if (
            $this->nullOnEmpty
            && (
                $this->value === null
                || empty(trim($this->value))
            )
        ) {
            return null;
        }
        if (array_key_exists(trim($this->value), $this->options)) {
            return $this->options[trim($this->value)]["value"];
        }
        return null;
    }

    /**
     * @param $value
     */
    public function serialize($value)
    {
        if ($value === null) {
            return;
        }

        $keyLabelFunction = $this->keyLabelFunction;
        if ($keyLabelFunction === null) {
            $key = $value;
        } else {
            $keyLabel = $keyLabelFunction($value, null);
            $key = $keyLabel["key"];
        }

        if (is_bool($key)) {
            $key = (int)$key;
        }
        if (array_key_exists($key, $this->options)) {
            $this->options[$key]["selected"] = true;
            $this->value = $key;
        }
    }


    public function getUnselectedText(): ?string
    {
        return $this->unselectedText;
    }

    /**
     * @param string $unselectedText
     * @return static
     */
    public function setUnselectedText(string $unselectedText)
    {
        $this->unselectedText = $unselectedText;
        return $this;
    }

    /**
     * @return bool
     */
    public function hasUnselectedText(): bool
    {
        return !empty($this->unselectedText);
    }

    /**
     * @return array
     */
    public function getOptions(): array
    {
        return $this->options;
    }

    /**
     *
     * @param mixed[] $options array of options if $keyLabelFunction is not provided array of arrays ['key' =>, 'label' => ] is expected
     * @param null|callable $keyLabelFunction callback function to extract data from options from $options parameter must return ['key' =>, 'label' => ] array, returned value is value of element from $options, can be changes by providing 'value'
     * @return static
     */
    public function setOptions(iterable $options, $keyLabelFunction = null)
    {
        if ($keyLabelFunction !== null) {
            if (!is_callable($keyLabelFunction)) { // @phpstan-ignore function.alreadyNarrowedType
                throw new Exception('$keyLabelFunction must be callable');
            }
            $this->keyLabelFunction = $keyLabelFunction;
        }
        $keyLabelFunction = $this->defaultKeyLabelFunction;
        if ($this->keyLabelFunction !== null) {
            $keyLabelFunction = $this->keyLabelFunction;
        }

        $this->options = [];
        $this->optionsStructure = [];
        foreach ($options as $index => $value) {
            $keyLabel = $keyLabelFunction($value, $index);
            if (!array_key_exists("key", $keyLabel)) {
                throw new Exception("\$keyLableFunction doesn't return key for option");
            }
            if (!array_key_exists("label", $keyLabel)) {
                throw new Exception("\$keyLableFunction doesn't return label for option");
            }
            if (array_key_exists("value", $keyLabel)) { // we pass value in
                $value = $keyLabel["value"];
            }
            $escape = true;
            if (array_key_exists("escape", $keyLabel)) { // we pass value in
                $escape = $keyLabel["escape"];
            }

            $this->options[$keyLabel["key"]] = [
                "value" => $value,
                "label" => $keyLabel["label"],
                'escape' => $escape,
                "selected" => false,
            ];
            $this->optionsStructure[] = [
                "type" => self::OPTION_TYPE,
                'key' => $keyLabel["key"],
            ];
        }

        return $this;
    }

    public function setKeyLabelFunction($keyLabelFunction)
    {
        if (!is_callable($keyLabelFunction)) {
            throw new Exception('$keyLabelFunction must be callable');
        }
        $this->keyLabelFunction = $keyLabelFunction;

        return $this;
    }

    public function addOption($option)
    {
        $keyLabelFunction = $this->defaultKeyLabelFunction;
        if ($this->keyLabelFunction !== null) {
            $keyLabelFunction = $this->keyLabelFunction;
        }

        $keyLabel = $keyLabelFunction($option, null);
        if (!array_key_exists("key", $keyLabel)) {
            throw new Exception("\$keyLableFunction doesn't return key for option");
        }
        if (!array_key_exists("label", $keyLabel)) {
            throw new Exception("\$keyLableFunction doesn't return label for option");
        }
        $value = null;
        if (array_key_exists("value", $keyLabel)) { // we pass value in
            $value = $keyLabel["value"];
        }
        $escape = true;
        if (array_key_exists("escape", $keyLabel)) { // we pass value in
            $escape = $keyLabel["escape"];
        }

        $this->options[$keyLabel["key"]] = [
            "value" => $value,
            "label" => $keyLabel["label"],
            'escape' => $escape,
            "selected" => false,
        ];
        $this->optionsStructure[] = [
            "type" => self::OPTION_TYPE,
            'key' => $keyLabel["key"],
        ];

        return $this;
    }

    /**
     * @return bool
     */
    public function hasOptions(): bool
    {
        return !empty($this->options);
    }

    /**
     * @param string $groupName
     * @param array $options
     * @return $this
     */
    public function addOptGroup(string $groupName, iterable $options)
    {
        $keyLabelFunction = $this->defaultKeyLabelFunction;
        if ($this->keyLabelFunction !== null) {
            $keyLabelFunction = $this->keyLabelFunction;
        }

        $optionKeys = [];

        foreach ($options as $index => $value) {
            $keyLabel = $keyLabelFunction($value, $index);
            if (!array_key_exists("key", $keyLabel)) {
                throw new Exception("\$keyLableFunction doesn't return key for option");
            }
            if (!array_key_exists("label", $keyLabel)) {
                throw new Exception("\$keyLableFunction doesn't return label for option");
            }
            if (array_key_exists("value", $keyLabel)) { // we pass value in
                $value = $keyLabel["value"];
            }
            $escape = true;
            if (array_key_exists("escape", $keyLabel)) { // we pass value in
                $escape = $keyLabel["escape"];
            }

            $this->options[$keyLabel["key"]] = [
                "value" => $value,
                "label" => $keyLabel["label"],
                'escape' => $escape,
                "selected" => false,
            ];
            $optionKeys[] = $keyLabel["key"];
        }

        $this->optionsStructure[] = [
            "type" => self::OPTGROUP_TYPE,
            "label" => $groupName,
            'optionKeys' => $optionKeys,
        ];

        return $this;
    }

    /**
     * returns options structure to be displayed
     * @return array[]
     */
    public function getOptionsStructure(): array
    {
        $structure = [];
        foreach ($this->optionsStructure as $optionStructure) {
            $struct = [
                'type' => $optionStructure['type'],
            ];
            if ($optionStructure['type'] == SelectField::OPTGROUP_TYPE) {
                $options = [];
                foreach ($optionStructure['optionKeys'] as $optionKey) {
                    $option = $this->options[$optionKey];
                    $options[] = [
                        'key' => $optionKey,
                        'selected' => $option['selected'],
                        'escape' => $option['escape'],
                        'label' => $option['label'],
                    ];
                }
                $struct['label'] = $optionStructure['label'];
                $struct["options"] = $options;
            } else {
                $option = $this->options[$optionStructure['key']];
                $struct['key'] = $optionStructure['key'];
                $struct['selected'] = $option['selected'];
                $struct['escape'] = $option['escape'];
                $struct['label'] = $option['label'];
            }
            $structure[] = $struct;
        }
        return $structure;
    }

    /**
     *
     */
    public function __clone()
    {
        // naklonuje validacni pravidla
        $clonedRules = [];
        foreach ($this->rules as $rule) {
            $clonedRule = clone $rule;
            $rule->setField($this);
            $clonedRules[] = $clonedRule;
        }
        $this->rules = $clonedRules;
    }

    protected function getSpecificSupportedProperties(): array
    {
        return [];
    }
}
