<?php

namespace IZON\Forms\Fields;

use InvalidArgumentException;
use IZON\Forms\ValidationRule;

/**
 * Pole odpovidajici rozsahu cisel ve kterem se da posunovat
 * https://www.w3.org/wiki/HTML/Elements/input/range
 */
class FloatRangeField extends BaseField
{
    /**
     * @var float minimalni hodnota, ktera muze byt do pole zadana
     */
    protected float $minValue;

    /**
     * @var float maximalni hodnota, ktera muze byt do pole zadana
     */
    protected float $maxValue;

    /**
     * @var float
     */
    protected float $step = 0.00000001;

    /**
     * @var array pole s moznymi hodnotami soupatoru
     */
    protected $valuesList = null;

    public function __construct(
        string $name,
        float $minValue,
        float $maxValue,
        string $label = ''
    ) {
        $this->name = $name;
        $this->label = $label;
        $this->propertyName = $this->name;

        if ($maxValue < $minValue) {
            throw new InvalidArgumentException("minValue = " . $minValue . " nesmi byt vetsi nez maxValue = " . $maxValue);
        }

        $this->minValue = $minValue;
        $this->maxValue = $maxValue;

        $rule = new ValidationRule();
        $rule->setMessage(_('Zadaná hodnota je mimo rozsah.'));

        $function = function (FloatRangeField $field, $args = []) {
            return !($field->getMinValue() > $field->parse() // je mensi nez minimum
                || $field->getMaxValue() < $field->parse()); // je vetsi nez maximum
        };
        $rule->setValidator($function);
        $this->addRule($rule);
    }

    public function getMinValue(): float
    {
        return $this->minValue;
    }

    public function setMinValue(float $minValue): FloatRangeField
    {
        $this->minValue = $minValue;
        return $this;
    }

    public function getMaxValue(): float
    {
        return $this->maxValue;
    }

    public function setMaxValue(float $maxPrice): FloatRangeField
    {
        $this->maxValue = $maxPrice;
        return $this;
    }

    public function parse()
    {
        return $this->value;
    }

    public function serialize($value)
    {
        $this->value = str_replace(",", ".", (string)$value);
    }

    public function getStep(): float
    {
        return $this->step;
    }

    public function setStep($step): FloatRangeField
    {
        if ($step <= 0) {
            throw new InvalidArgumentException("Step nesmi byt nulovy nebo zaporny");
        }
        $this->step = $step;
        return $this;
    }

    protected function getSpecificSupportedProperties(): array
    {
        return [];
    }
}
