<?php

namespace IZON\Forms\Fields;

use Exception;
use IZON\Forms\ValidationRule;

/**
 * Zaklad fomularoveho pole
 */
abstract class BaseField
{
    /**
     * properties supported for all html field types
     */
    public const COMMON_SUPPORTED_PROPERTIES = [
        'readonly',
        'disabled',
        'novalidate',
    ];

    /**
     * @var string[] error messages to be displayed in the form
     */
    protected array $errors = [];

    /**
     * true, pokud musi byt pritomne ve formulari
     * @var bool
     */
    protected bool $required = false;

    /**
     * Ma se pouzit label?
     * Defaultne true
     *
     * @var bool
     *
     * @deprecated jestli pouzit label se nastavuji ve view
     */
    protected bool $useLabel = true;

    /**
     * popiska formularoveho pole
     * @var string
     */
    protected string $label;

    /**
     * stylova trida popisky formularoveho pole
     * @var null|string
     *
     * @deprecated css tridy a se nastavuji ve view
     */
    protected ?string $labelClass;

    /**
     * relativni nazev pole v danem formulari
     * podle ktereho se bude ziskavat hornota z request
     * @var string
     */
    protected string $name;

    /**
     * jaky prefix se ma pripojit pred name ve formulari
     * @var string[]
     */
    protected array $prefix = [];

    /**
     * nazev do ktere se ma hodnota formularoveho prvku zkopirovat
     * defaultne se nastavuje jako name
     *
     * v pripade ze je zanoreno do podobjektu tak se predava jako pole se jmeny zleva doprava
     * @var null|string
     */
    protected ?string $propertyName = null;

    /**
     * Pole s atributy tagu input, napr.: placeholder, readonly, selected atp.
     *
     * @var array<string, int|string|array>
     */
    protected array $attributes = [];

    /**
     * @var string[] properties like required, disabled, ...
     */
    protected $properties = [];

    /**
     * hodnota ziskana z formulare pomoci POST, GET nebo RESULT pole
     * a takova, ktera se ma do html formulare vlozit
     * @var mixed
     */
    protected $value;

    /**
     * defaultni hodnota pole
     * @var mixed
     */
    protected $default;

    /**
     * validacni pravidla
     * @var ValidationRule[]
     */
    protected array $rules = [];

    /**
     * @var bool pokud je pole prazdne vloz null
     */
    protected bool $nullOnEmpty = false;

    /**
     * @var bool if false data from field is not flushed back to object
     */
    protected bool $oneWay = false;

    /**
     * vraci hodnotu primo odpovidajici parametru ziskaneho z http
     */
    public function getValue()
    {
        return $this->value;
    }

    /**
     * inicializuje pole hodnotou zislanou z POST, GET, REQUEST
     * @param mixed $value
     * @return static
     */
    public function setValue($value)
    {
        $this->value = $value;
        return $this;
    }

    /**
     * performs validation on the field
     */
    public function validate(): void
    {
        $this->errors = [];
        if (
            $this->required
            && $this->value == null) {
            $this->errors[] = _("Pole je povinné");
        }
        foreach ($this->rules as $rule) {
            if (!$rule->validate($this)) {
                $this->errors[] = sprintf($rule->getMessage(), $rule->getArgs());
            }
        }
    }

    public function addError(string $error): void
    {
        $this->errors[] = $error;
    }

    /**
     * @return string[] returns all errors found during validation
     */
    public function getErrors(): array
    {
        return $this->errors;
    }

    /**
     * @return bool true if validation found some errors
     */
    public function hasErrors(): bool
    {
        return !empty($this->errors);
    }

    /**
     * clears all form errors
     */
    public function clearErrors()
    {
        $this->errors = [];
    }

    /**
     * parsuje vstup z formulaze do vnitrni podoby pouzivane v php
     * napriklad string date do objektu date
     * vraci tuto vytvorenou podobu
     * @return mixed
     */
    abstract public function parse();

    /**
     * provede serializaci hodnoty do tvaru pouzitelneho v html formulari
     * @param mixed $value
     * @return void
     */
    abstract public function serialize($value);

    public function getLabel()
    {
        return $this->label;
    }

    /**
     * @param string $label
     * @param null|string $class - bude odstraneno, lable class se nastavuje ve view
     * @return static
     */
    public function setLabel(string $label, ?string $class = null)
    {
        $this->label = $label;
        $this->labelClass = $class; // @phpstan-ignore-line property.deprecated
        return $this;
    }

    public function getLabelClass(): ?string
    {
        return $this->labelClass;  // @phpstan-ignore-line property.deprecated
    }

    /**
     * @param string $labelClass
     * @return static
     * @deprecated css tridy a se nastavuji ve view
     */
    public function setLabelClass(string $labelClass)
    {
        $this->labelClass = $labelClass;
        return $this;
    }

    public function getName()
    {
        return $this->name;
    }


    public function setName(string $name)
    {
        $this->name = $name;
        return $this;
    }

    /**
     * jak se ma jenovat pri vypisovani formulare, aby se spravne
     * navazalo pole _REQUEST
     * @return string
     */
    public function getFormName()
    {
        if (
            $this->prefix
            && !empty($this->prefix)
        ) {
            $ret = "";
            foreach ($this->prefix as $key => $value) {
                if ($key == 0) {
                    $ret .= $value;
                } else {
                    $ret .= "[" . $value . "]";
                }
            }
            return $ret . "[" . $this->name . "]";
        }
        return $this->name;
    }

    /**
     * nastavi prefix
     * @param string[] $prefix
     */
    public function setPrefix(array $prefix)
    {
        $this->prefix = $prefix;
        return $this;
    }

    public function getPropertyName()
    {
        return $this->propertyName;
    }

    public function setPropertyName(string $propertyName)
    {
        $this->propertyName = $propertyName;
        return $this;
    }

    public function getRules()
    {
        return $this->rules;
    }

    /**
     * @deprecated use isRequired
     */
    public function getRequired(): bool
    {
        return $this->isRequired();
    }

    public function isRequired(): bool
    {
        return $this->required;
    }

    public function setRequired($required = true)
    {
        $this->required = $required;
        return $this;
    }

    public function addRule(ValidationRule $rule)
    {
        $rule->setField($this);
        $this->rules[] = $rule;
        return $this;
    }

    public function getDefault()
    {
        return $this->default;
    }

    public function setDefault($default): self
    {
        $this->default = $default;
        return $this;
    }

    public function getAttributes()
    {
        return $this->attributes;
    }

    public function hasAttribute($name)
    {
        return isset($this->attributes[$name]);
    }

    public function getAttribute($name)
    {
        if (!isset($this->attributes[$name])) {
            return null;
        }
        return $this->attributes[$name];
    }

    /**
     * Vrati atributy pole pro vypis do html
     * @return string
     * @deprecated should be handled by template rendering system
     */
    public function getAttributesAsString(): string
    {
        $ret = [];
        foreach ($this->attributes as $key => $value) {
            if ($key == 'class') { // property class obsahuje pole trid
                $value = implode(" ", $value);
            }
            $ret[] = $key . '="' . $value . '"';
        }
        if ($this->required) {
            $ret[] = 'required="required"';
        }
        return implode(' ', $ret);
    }

    /**
     * Nastavuje jednotlive atributy pro input.<br>
     * Pro class a id jsou udelane samostane metody.
     *
     * @param string $key
     * @param string|number $value
     * @return static
     */
    public function addAttribute(string $key, $value)
    {
        $reject = ['class', 'id'];
        if (in_array($key, $reject)) {
            throw new Exception('For atributes ' . strtoupper(implode(', ', $reject)) . ' use their own setters.');
        }
        if (in_array($key, $this->getSupprtedProperties())) {
            // change to E_WARNING in later version
            trigger_error('For properties use addProperty instead.');
        }

        $this->attributes[$key] = $value;
        return $this;
    }

    /**
     * @return string[] all properties that field supports
     */
    private function getSupprtedProperties(): array
    {
        return array_merge(self::COMMON_SUPPORTED_PROPERTIES, $this->getSpecificSupportedProperties());
    }

    /**
     * @return string[] returns propertied that are specific for field type
     */
    abstract protected function getSpecificSupportedProperties(): array;

    /**
     * @return string[]
     */
    public function getProperties(): array
    {
        return array_keys($this->properties);
    }

    /**
     * removes property from field
     * @param string $propName
     * @return static
     */
    public function removeProperty($propName)
    {
        unset($this->properties[$propName]);
        return $this;
    }

    /**
     * @return bool true if field property readonly
     */
    public function isReadonly(): bool
    {
        return $this->hasProperty("readonly");
    }

    /**
     * sets or removes property readonly to/from field
     * @param bool $readonly
     * @return static
     */
    public function setReadonly($readonly = true)
    {
        $this->addProperty("readonly");
        return $this;
    }

    /**
     *
     * @param string $name
     * @return bool
     */
    public function hasProperty($name): bool
    {
        return isset($this->properties[$name]);
    }

    /**
     * adds property to filed
     * @param string $propName
     * @return static
     */
    public function addProperty(string $propName)
    {
        if (!in_array($propName, $this->getSupprtedProperties())) {
            throw new Exception('Property ' . $propName . ' is not supported for this field type.');
        }
        $this->properties[$propName] = $propName;
        return $this;
    }

    public function isOneWay(): bool
    {
        return $this->oneWay;
    }

    /**
     * @param boolean $oneWay pokud je true, tak se nema hodnota z formulare zpet nastavovat do objektu pomoci flush
     */
    public function setOneWay(bool $oneWay = true)
    {
        $this->oneWay = $oneWay;
        return $this;
    }

    /**
     * Prida stylovou tridu
     * @param string $class - je string s nazvem tridy, ktera se ma pridat
     * @return static
     * @deprecated css tridy a se nastavuji ve view
     */
    public function addClass(string $class)
    {
        if (array_key_exists('class', $this->attributes)) {
            $param = [$class];
            $this->attributes['class'] = array_unique(array_merge($this->attributes['class'], $param));
        } else {
            $this->attributes['class'] = [$class];
        }
        return $this;
    }

    /**
     * Prida stylove tridy
     * @param mixed $param - bud pole stylovych trid, nebo string trid oddelenych mezerou, ie: 'trida01 trida02 trida03'
     * @return BaseField
     * @deprecated css tridy a se nastavuji ve view
     */
    public function addClasses($param)
    {
        if (!is_array($param)) {
            $param = explode(' ', trim($param));
        }
        if (array_key_exists('class', $this->attributes)) {
            $this->attributes['class'] = array_unique(array_merge($this->attributes['class'], $param));
        } else {
            $this->attributes['class'] = $param;
        }
        return $this;
    }

    /**
     * Nastavi id prvku
     * @param string $param
     * @return BaseField
     * @deprecated css tridy a se nastavuji ve view
     */
    public function setId($param)
    {
        $this->attributes['id'] = $param;
        return $this;
    }

    /**
     * Vrati id prvku
     * @return mixed [string|null]
     * @deprecated css tridy a se nastavuji ve view
     */
    public function getId()
    {
        if (isset($this->attributes['id'])) {
            return $this->attributes['id'];
        }
        return null;
    }

    /**
     * Ma se vypsat label
     * @return boolean
     * @deprecated css tridy a se nastavuji ve view
     */
    public function getUseLabel()
    {
        return $this->useLabel;
    }

    /**
     * @param boolean $useLabel
     * @return static
     * @deprecated  css tridy a se nastavuji ve view
     */
    public function setUseLabel($useLabel)
    {
        $this->useLabel = $useLabel;
        return $this;
    }

    /**
     * pokud true tak se prazdny retezec meni na null
     * @param bool $nullOnEmpty
     * @return static
     */
    public function setNullOnEmpty(bool $nullOnEmpty)
    {
        $this->nullOnEmpty = $nullOnEmpty;
        return $this;
    }

    public function __clone()
    {
        // rests errors
        $this->errors = [];
        // clone validation validation rules
        $clonedRules = [];
        foreach ($this->rules as $rule) {
            $clonedRule = clone $rule;
            $rule->setField($this);
            $clonedRules[] = $clonedRule;
        }
        $this->rules = $clonedRules;
    }
}
